package org.ow2.orchestra.cxf;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.wsdl.Definition;
import javax.wsdl.Import;
import javax.wsdl.OperationType;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.wsdl.Types;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.wsdl.extensions.schema.Schema;
import javax.wsdl.extensions.schema.SchemaImport;
import javax.wsdl.extensions.schema.SchemaReference;
import javax.wsdl.extensions.soap.SOAPBinding;
import javax.wsdl.extensions.soap.SOAPBody;
import javax.wsdl.extensions.soap.SOAPFault;
import javax.wsdl.extensions.soap12.SOAP12Binding;
import javax.wsdl.extensions.soap12.SOAP12Body;
import javax.wsdl.extensions.soap12.SOAP12Fault;
import javax.xml.namespace.QName;

import org.apache.cxf.Bus;
import org.apache.cxf.BusFactory;
import org.apache.cxf.common.util.PackageUtils;
import org.apache.cxf.endpoint.Server;
import org.apache.cxf.endpoint.ServerRegistry;
import org.apache.cxf.frontend.ServerFactoryBean;
import org.apache.cxf.service.factory.ReflectionServiceFactoryBean;
import org.apache.cxf.service.model.OperationInfo;
import org.apache.cxf.wsdl11.WSDLServiceFactory;
import org.ow2.orchestra.definition.BpelProcess;
import org.ow2.orchestra.facade.Deployment;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.XmlConstants;
import org.ow2.orchestra.util.XmlUtil;
import org.ow2.orchestra.util.wsdl.WsdlUtil;
import org.ow2.orchestra.ws.WSDeployer;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * CXF engine WS deployer.
 *
 * @author Guillaume Porcher
 */
public class CxfDeployer extends WSDeployer {

  public static final Bus CXF_BUS = BusFactory.getDefaultBus();

  public static final String WSDL_FILE_NAME = ".wsdl";

  /** log. */
  private static Logger log = Logger.getLogger(Deployment.class.getName());

  private static File webAppTmpDir;

  private final File servicesDir;

  private final BpelProcess bpelProcess;

  /**
   * Default constructor.
   *
   * @param wsId - id of the WS : in our case, processName
   * @param wsdlUrl - url of the wsdl file to deploy.
   * @param orchestraDirectory - Absolute path to the orchestra directory.
   */
  public CxfDeployer(final BpelProcess bpelProcess) {
    super(bpelProcess.getUUID());
    this.bpelProcess = bpelProcess;
    this.servicesDir = this.getServicesDir();
  }

  /**
   * return null if operation style is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param operationStyle - operationStyle to check
   * @return null if operation style is supported by Axis engine. Else, returns
   *         a message explaining the problem.
   */
  @Override
  protected String checkOperationStyle(final String operationStyle) {
    if (operationStyle == null) {
      return "Style attribute of this operation must be specified";
    } else if (!operationStyle.equals("document") && !operationStyle.equals("rpc")) {
      return "Style attribute of this operation must be : document or rpc";
    }
    return null;
  }

  /**
   * return null if operation type is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param operationType - operationType to check
   * @return null if operation type is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  @Override
  protected String checkOperationType(final OperationType operationType) {
    if (!operationType.equals(OperationType.REQUEST_RESPONSE) && !operationType.equals(OperationType.ONE_WAY)) {
      return "Operation type : "
        + operationType + " is not supported. Please use one of : "
        + OperationType.ONE_WAY + "/" + OperationType.REQUEST_RESPONSE;
    }
    return null;
  }

  /**
   * return null if soapVersion is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param soapBinding - soapBinding to check
   * @return null if soapVersion is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  @Override
  protected String checkSoapVersion(final ExtensibilityElement soapBinding) {
    if (!(soapBinding instanceof SOAPBinding) && !(soapBinding instanceof SOAP12Binding)) {
      return "Supported Soap Version are " + WSDeployer.URI_WSDL11_SOAP11 + "/" + WSDeployer.URI_WSDL11_SOAP12;
    }
    return null;
  }

  /**
   * return null if transport is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param soapBinding - soapBinding to check
   * @return null if transport is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  @Override
  protected String checkTransport(final ExtensibilityElement soapBinding) {
    String transportUri = "";
    if (soapBinding instanceof SOAPBinding) {
      transportUri = ((SOAPBinding) soapBinding).getTransportURI();
    } else if (soapBinding instanceof SOAP12Binding) {
      transportUri = ((SOAP12Binding) soapBinding).getTransportURI();
    }
    if (!WSDeployer.SOAP_HTTP_TRANSPORT_URI.equals(transportUri)) {
      return "Transport URI : " + transportUri + " is not supported. Please use " + WSDeployer.SOAP_HTTP_TRANSPORT_URI;
    }
    return null;
  }

  /**
   * return null if use is supported by Axis engine. Else, returns a message
   * explaining the problem.
   *
   * @param soapBody - soapBody to check
   * @return null if use is supported by Axis engine. Else, returns a message
   *         explaining the problem.
   */
  @Override
  protected String checkUse(final ExtensibilityElement element) {
    String use = "";
    if (element instanceof SOAPBody) {
      use = ((SOAPBody) element).getUse();
    } else if (element instanceof SOAP12Body) {
      use = ((SOAP12Body) element).getUse();
    } else if (element instanceof SOAPFault) {
      use = ((SOAPFault) element).getUse();
    } else if (element instanceof SOAP12Fault) {
      use = ((SOAP12Fault) element).getUse();
    }
    if (!"literal".equals(use)) {
      return "Use : " + use + " is not supported. Please use " + "literal";
    }
    return null;
  }

  /**
   * return null if soapBody attributes are supported by Axis engine. Else,
   * returns a message explaining the problem.
   *
   * @param soapBody - soapBody to check
   * @return null if soapBody attributes are supported supported by Axis engine.
   *         Else, returns a message explaining the problem.
   */
  @Override
  protected String checkSoapBody(final ExtensibilityElement soapBody) {
    List parts = null;
    if (soapBody instanceof SOAPBody) {
      parts = ((SOAPBody) soapBody).getParts();
    } else if (soapBody instanceof SOAP12Body) {
      parts = ((SOAP12Body) soapBody).getParts();
    }
    if (parts != null) {
      return "SoapBody is using parts attribute which is not currently supported.";
    }
    return null;
  }

  @Override
  protected String checkSoapFault(final ExtensibilityElement soapFault) {
    String name = null;
    if (soapFault instanceof SOAPFault) {
      name = ((SOAPFault) soapFault).getName();
    } else if (soapFault instanceof SOAP12Fault) {
      name = ((SOAP12Fault) soapFault).getName();
    }
    if (name == null) {
      return "SoapFault is not specifying fault name which is not currently supported.";
    }
    return null;
  }



  /**
   * This method will deploy the specified WS Service on the choosen ws engine.
   * This method must be overriden by each ws engine Deployer.
   *
   * @param def - Definition object that represents a WS Service to deploy.
   */
  @Override
  protected void deployServices(final List<Service> services) {
    for (final Service service : services) {
      final File wsdlFile = this.createWsdlFile(service);

      final QName serviceQName = service.getQName();

      final Map<String, Port> ports = service.getPorts();
      for (final Port port : ports.values()) {

        final QName endpointQName = new QName(serviceQName.getNamespaceURI(), port.getName());

        final ServerFactoryBean svrBean = new ServerFactoryBean();

        // set cxf bus
        svrBean.setBus(CxfDeployer.CXF_BUS);

        // set address of the WS
        final String endpointURL = this.getAddressFromPort(port);
        if (endpointURL != null) {
          svrBean.setAddress(endpointURL);
        } else {
          svrBean.setAddress("/" + port.getName());
        }

        // set service name
        svrBean.setServiceName(serviceQName);

        // set service wsdl url
        try {

          final WSDLServiceFactory wsdlServiceFactory =
            new WSDLServiceFactory(CxfDeployer.CXF_BUS, wsdlFile.toURI().toURL().toExternalForm(), serviceQName);
          wsdlServiceFactory.setEndpointName(endpointQName);

          final org.apache.cxf.service.Service cxfService = wsdlServiceFactory.create();

          // unset wrapped attribute in operation: we want cxf to return the elements as defined in the wsdl
          for (final OperationInfo opInfo : cxfService.getEndpointInfo(endpointQName).getInterface().getOperations()) {
            opInfo.setUnwrappedOperation(null);
          }

          final QName portTypeQname = port.getBinding().getPortType().getQName();
          final ReflectionServiceFactoryBean rsfb =
            new OrchestraReflectionServiceFactoryBean(portTypeQname, cxfService);
          rsfb.setServiceName(serviceQName);
          rsfb.setEndpointName(endpointQName);

          rsfb.setInvoker(new CxfWSImpl(port.getBinding().getPortType(), this.processUUID));

          svrBean.setServiceFactory(rsfb);

        } catch (final MalformedURLException e) {
          throw new OrchestraRuntimeException(e);
        }

        // create server
        svrBean.setStart(false);
        final Server server = svrBean.create();

        // setup interceptors
        CxfUtils.setUpOrchestraInterceptors(server.getEndpoint());

        // start
        server.start();
      }
    }
  }

  /**
   * This method will undeploy the specified WS Service from the choosen ws
   * engine. This method must be overriden by each ws engine Deployer.
   *
   * @param def - Definition object that represents a WS Service to undeploy.
   */
  @Override
  protected void undeployServices(final List<Service> services) {
    final ServerRegistry serverRegistry = CxfDeployer.CXF_BUS.getExtension(ServerRegistry.class);

    for (final Service service : services) {
      final List<Server> servers = serverRegistry.getServers();
      if (servers != null) {
        for (final Server server : new ArrayList<Server>(servers)) {
          if (server.getEndpoint().getService().getName().equals(service.getQName())) {
            server.stop();
          }
        }
      }
    }
  }



  /**
   * Return the deploy wsdl file of the given definition.
   *
   * @param def : def to analyse
   * @return the deploy wsdl file used by the given def
   */
  private File getWsdlFile(final QName serviceQName) {
    return new File(
        this.getWsdlFilesDir(serviceQName.getNamespaceURI())
        + File.separator + serviceQName.getLocalPart() + CxfDeployer.WSDL_FILE_NAME);
  }


  /**
   * Returns the directory where wsdl files of this definition could be found.
   *
   * @param def : def to analyse
   * @return the directory where wsdl files of this definition could be found
   */
  private String getWsdlFilesDir(final String wsdlTargetnamespace) {
    final String wsdlFilesDir =
      this.servicesDir
      + File.separator
      + CxfDeployer.getDirectoryFromPackage(
          CxfDeployer.getPackageFromNamespace(wsdlTargetnamespace));
    return wsdlFilesDir;
  }



  private static synchronized File getWStempDir() {
    if (CxfDeployer.webAppTmpDir == null) {
      try {
        final File tmpDir = new File(System.getProperty("java.io.tmpdir"));
        if (!tmpDir.exists() && !tmpDir.mkdirs()) {
          throw new IOException("Cannot create the temporary directory '" + tmpDir.getAbsolutePath() + "'.");
        }
        CxfDeployer.webAppTmpDir = File.createTempFile("orchWS-Repo-", null, null);
        CxfDeployer.webAppTmpDir.delete();
        if (!CxfDeployer.webAppTmpDir.mkdirs()) {
          throw new IOException("Cannot create the temporary directory '" + CxfDeployer.webAppTmpDir + "'.");
        }
        if (CxfDeployer.log.isLoggable(Level.FINE)) {
          CxfDeployer.log.fine("webAppTmpDir created : " + CxfDeployer.webAppTmpDir.getAbsolutePath());
        }
      } catch (final Exception e) {
        throw new OrchestraRuntimeException("Error creating " + CxfDeployer.webAppTmpDir, e);
      }
    }
    return CxfDeployer.webAppTmpDir;
  }

  synchronized void resetServiceDir() {
    if (this.servicesDir != null) {
      Misc.deleteDir(this.servicesDir);
    }
  }

  private File getServicesDir() {
    final String packag = CxfDeployer.getPackageFromNamespace(this.bpelProcess.getName());
    return new File(CxfDeployer.getWStempDir() + File.separator + packag + "__" + this.bpelProcess.getName());
  }


  /**
   * Returns the java package that maps to the given namespace.
   *
   * @param ns - ns
   * @return the java package that maps to the given namespace.
   */
  public static String getPackageFromNamespace(final String ns) {
    final String packag = PackageUtils.getPackageNameByNameSpaceURI(ns);
    return packag;
  }

  /**
   * Returns the directory structure corresponding to the given package.
   *
   * @param packag - packag
   * @return the directory structure corresponding to the given package.
   */
  public static String getDirectoryFromPackage(final String packag) {
    final String dir = packag.replace('.', File.separatorChar);
    return dir;
  }



  /**
   * Generate a wsdl file for the definition.
   * Generate files for all imported resources of this wsdl definition.
   * @param wsdlDefinition
   * @return
   */
  private File createWsdlFile(final Definition wsdlDefinition, final File wsdlDir) {
    try {
      // Process wsdl imports
      final Map<String, List<Import>> imports = wsdlDefinition.getImports();
      if (!imports.isEmpty()) {
        for (final List<Import> l : imports.values()) {
          for (final Import impor : l) {
            final Definition importedDefinition = impor.getDefinition();
            final File importFile = this.createWsdlFile(importedDefinition, wsdlDir);
            impor.setLocationURI(importFile.getName());
          }
        }
      }
      // Process XML Schema imports
      final Types types = wsdlDefinition.getTypes();
      if (types != null) {
        for (final ExtensibilityElement elt : (List<ExtensibilityElement>) types.getExtensibilityElements()) {
          if (elt instanceof Schema) {
            final Schema schema = (Schema) elt;
            this.createImportedXsds(wsdlDir, schema);
          }
        }
      }
      // Write WSDL file
      final File wsdlFile = File.createTempFile("generated", ".wsdl", wsdlDir);
      final FileOutputStream fos = new FileOutputStream(wsdlFile);
      try {
        WsdlUtil.writeWsdl(wsdlDefinition, fos);
      } finally {
        fos.close();
      }
      return wsdlFile.getAbsoluteFile();
    } catch (final IOException e) {
      throw new OrchestraRuntimeException(e);
    }
  }

  /**
   * Creates XSD files for all schemas imploted by the xml schema
   * @param wsdlDir
   * @param schema
   */
  private void createImportedXsds(final File wsdlDir, final Schema schema) {
    // process xsd:import
    final NodeList importsNl =
      schema.getElement().getOwnerDocument().getElementsByTagNameNS(XmlConstants.XMLNS_XMLSCHEMA, "import");
    for (int i = 0; i < importsNl.getLength(); i++) {
      final Element importElt = (Element) importsNl.item(i);
      if (importElt.hasAttribute("schemaLocation")) {
        final Map<String, List<SchemaImport>> schemaImportsMap = schema.getImports();
        final List<SchemaImport> schemaImports = schemaImportsMap.get(importElt.getAttribute("namespace"));
        for (final SchemaImport schemaImport : schemaImports) {
          if (schemaImport.getSchemaLocationURI() != null
              && schemaImport.getSchemaLocationURI().equals(importElt.getAttribute("schemaLocation"))) {
            final File refFile = this.createXsdFile(schemaImport.getReferencedSchema(), wsdlDir);
            schemaImport.setSchemaLocationURI(refFile.toURI().toString());
            importElt.setAttribute("schemaLocation", refFile.toURI().toString());
          }
        }
      }
    }
    // process xsd:include
    final NodeList includeNl =
      schema.getElement().getOwnerDocument().getElementsByTagNameNS(XmlConstants.XMLNS_XMLSCHEMA, "include");
    for (int i = 0; i < includeNl.getLength(); i++) {
      final Element includeElt = (Element) includeNl.item(i);
      if (includeElt.hasAttribute("schemaLocation")) {
        for (final SchemaReference schemaReference : (List<SchemaReference>) schema.getIncludes()) {
          if (schemaReference.getSchemaLocationURI() != null
              && schemaReference.getSchemaLocationURI().equals(includeElt.getAttribute("schemaLocation"))) {
            final File refFile = this.createXsdFile(schemaReference.getReferencedSchema(), wsdlDir);
            schemaReference.setSchemaLocationURI(refFile.toURI().toString());
            includeElt.setAttribute("schemaLocation", refFile.toURI().toString());
          }
        }
      }
    }
    // process xsd:redefines
    final NodeList redefinesNl =
      schema.getElement().getOwnerDocument().getElementsByTagNameNS(XmlConstants.XMLNS_XMLSCHEMA, "redefine");
    for (int i = 0; i < redefinesNl.getLength(); i++) {
      final Element redefineElt = (Element) redefinesNl.item(i);
      if (redefineElt.hasAttribute("schemaLocation")) {
        for (final SchemaReference schemaReference : (List<SchemaReference>) schema.getRedefines()) {
          if (schemaReference.getSchemaLocationURI() != null
              && schemaReference.getSchemaLocationURI().equals(redefineElt.getAttribute("schemaLocation"))) {
            final File refFile = this.createXsdFile(schemaReference.getReferencedSchema(), wsdlDir);
            schemaReference.setSchemaLocationURI(refFile.toURI().toString());
            redefineElt.setAttribute("schemaLocation", refFile.toURI().toString());
          }
        }
      }
    }
  }

  /**
   * Create a XSD file for this xml schema and all imported schemas.
   * @param schema
   * @param wsdlDir
   * @return
   */
  private File createXsdFile(final Schema schema, final File wsdlDir) {
    this.createImportedXsds(wsdlDir, schema);
    try {
      final File xsdFile = File.createTempFile("generated", ".xsd", wsdlDir);
      final Element schemaElement = schema.getElement();
      XmlUtil.writeXmlFile(schemaElement.getOwnerDocument(), xsdFile);
      return xsdFile;
    } catch (final IOException e) {
      throw new OrchestraRuntimeException(e);
    }
  }

  /**
   * Generate a wsdl file for the service
   * @param service
   * @return
   */
  private File createWsdlFile(final Service service) {
    final File wsdlFile = this.getWsdlFile(service.getQName());
    final File wsdlDir = wsdlFile.getParentFile();
    wsdlDir.mkdirs();

    final Definition wsdlDefinition = this.bpelProcess.getWsdlInfos().extractServiceWsdlDefinition(service);

    return this.createWsdlFile(wsdlDefinition, wsdlDir);
  }

  /* (non-Javadoc)
   * @see org.ow2.orchestra.ws.WSDeployer#checkWSEngineIsAvailable()
   */
  @Override
  protected void checkWSEngineIsAvailable() {
    // TODO
  }


}
