package org.ow2.orchestra.cxf;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.logging.Level;

import javax.servlet.Servlet;
import javax.wsdl.Service;
import javax.xml.namespace.QName;

import org.apache.camel.CamelContext;
import org.apache.camel.spring.SpringCamelContext;
import org.apache.cxf.Bus;
import org.ops4j.pax.swissbox.tinybundles.core.BuildableBundle;
import org.ops4j.pax.swissbox.tinybundles.core.TinyBundle;
import org.ops4j.pax.swissbox.tinybundles.core.TinyBundles;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceRegistration;
import org.ow2.orchestra.definition.BpelProcess;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.facade.uuid.ProcessDefinitionUUID;
import org.ow2.orchestra.osgi.OrchestraOSGiEngine;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.services.impl.PublisherImpl;
import org.ow2.orchestra.services.itf.HttpPublisher;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.OrchestraConstants;
import org.ow2.orchestra.util.ProcessResourcesRepository;
import org.ow2.orchestra.util.XmlUtil;
import org.springframework.context.ApplicationContext;
import org.springframework.osgi.context.event.OsgiBundleApplicationContextEvent;
import org.springframework.osgi.context.event.OsgiBundleApplicationContextListener;
import org.springframework.osgi.context.event.OsgiBundleContextFailedEvent;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * cxfServicePublisher.java. Implementation of the service publisher with cxf.
 *
 * @author Guillaume Porcher
 *
 */
public class CxfPublisher extends PublisherImpl implements HttpPublisher {

  /**
   * @author Guillaume Porcher
   *
   */
  private static final class BpelBundleApplicationContextListener implements OsgiBundleApplicationContextListener {

    /**
     *
     */
    private final Bundle installedBundle;
    private Throwable failure;
    private ApplicationContext applicationContext;

    /**
     * @param installedBundle
     */
    private BpelBundleApplicationContextListener(final Bundle installedBundle) {
      this.installedBundle = installedBundle;
    }
    public void onOsgiApplicationEvent(final OsgiBundleApplicationContextEvent event) {
      if (event.getBundle().equals(this.installedBundle)) {
        if (event instanceof OsgiBundleContextFailedEvent) {
          this.failure = ((OsgiBundleContextFailedEvent) event).getFailureCause();
        }
        this.applicationContext = event.getApplicationContext();
      }
    }
    /**
     * @return the failure
     */
    public Throwable getFailure() {
      return this.failure;
    }


    /**
     * @return the applicationContext
     */
    public ApplicationContext getApplicationContext() {
      return this.applicationContext;
    }

  }

  private final Map<QName, SpringCamelContext> camelContexts =
    new HashMap<QName, SpringCamelContext>();
  private final Map<QName, Bundle> bundles =
    new HashMap<QName, Bundle>();

  private Bus cxfBus;

  /**
   * @return the bundleContext
   */
  public BundleContext getBundleContext() {
    return (BundleContext) Environment.getFromCurrent(OrchestraOSGiEngine.ORCHESTRA_BUNDLE_CONTEXT_KEY);
  }

  /**
   * Create camel context for the process.
   * @param bpelProcess
   */
  private void createCamelContext(final BpelProcess bpelProcess) {
    // get camel-context file from process resources
    final byte[] camelContextBytes = bpelProcess.getResourcesRepository().getResources().get("camel-context.xml");
    if (camelContextBytes != null) {
      try {
        // create bundle
        final InputStream processBundleStream = this.createBundle(bpelProcess);
        // deploy bundle
        final ApplicationContext applicationContext = this.deployBundle(processBundleStream, bpelProcess);
        // get camel context from bundle application context
        final SpringCamelContext camelContext = SpringCamelContext.springCamelContext(applicationContext);
        // Store context in map
        this.camelContexts.put(bpelProcess.getQName(), camelContext);
      } catch (final Exception e) {
        throw new OrchestraRuntimeException("Error while creating camel context", e);
      }
    }
  }

  private void deleteCamelContext(final BpelProcess bpelProcess) {
    final Bundle bundle = this.bundles.remove(bpelProcess.getQName());
    this.camelContexts.remove(bpelProcess.getQName());
    if (bundle != null) {
      try {
        bundle.uninstall();
      } catch (final BundleException e) {
        throw new OrchestraRuntimeException(e);
      }

    }
  }

  public CamelContext getCamelContext(final QName processQName) {
    return this.camelContexts.get(processQName);
  }


  @Override
  public void publishServices(final BpelProcess bpelProcess, final Environment environment) {
    super.publishServices(bpelProcess, environment);

    this.doDeploy(bpelProcess);
  }

  /**
   * Create an OSGI bundle from the bpel process resources.
   * Instrument the bundle so that the camel-context.xml is processed by spring-osgi-extender.
   * @param bpelProcess
   * @return
   */
  public InputStream createBundle(final BpelProcess bpelProcess) {
    // Get process resources
    final ProcessResourcesRepository processResourcesRepository = bpelProcess.getResourcesRepository();

    // Create bundle content from resources
    // camel-context.xml is moved to META-INF/spring for bnd instrumentation
    final TinyBundle bundle = TinyBundles.newBundle();
    for (final Entry<String, byte[]> entry : processResourcesRepository.getResources().entrySet()) {
      if (!entry.getKey().equals("camel-context.xml")) {
        bundle.add(entry.getKey(), new ByteArrayInputStream(entry.getValue()));
      } else {
        bundle.add("META-INF/spring/camel-context.xml",
          new ByteArrayInputStream(
            this.instrumentContextDefinition(entry.getValue(), bpelProcess.getUUID())
          )
        );
      }
    }
    // Prepare bundle Manifest
    // Use spring plugin to add imports for classes used in camel-context.xml
    final BuildableBundle buildableBundle = bundle.prepare(
        TinyBundles.withBnd()
        .set(Constants.BUNDLE_SYMBOLICNAME, "Orchestra process: " + bpelProcess.getUUID())
        .set("Spring-Context", "META-INF/spring/camel-context.xml;create-asynchronously:=false")
        .set("-plugin", "aQute.lib.spring.SpringComponent")
    );
    // Create bundle as stream
    final InputStream bundleStream = buildableBundle.build();
    return bundleStream;
  }

  /**
   * @param value
   * @return
   */
  private byte[] instrumentContextDefinition(final byte[] value, final ProcessDefinitionUUID processDefinitionUUID) {
    final Document doc = XmlUtil.getDocumentFromString(new String(value));
    final NodeList nl = doc.getElementsByTagNameNS("http://camel.apache.org/schema/spring", "camelContext");
    if (nl != null) {
      for (int i = 0; i < nl.getLength(); i++) {
        final Element el = (Element) nl.item(i);

        el.setAttribute("autoStartup", "true");
        final String oldId = XmlUtil.attribute(el, "id");
        final String newId = "orchestra-" + processDefinitionUUID + (i == 0 ? "" : "-" + i);
        if (oldId != null) {
          final Element alias = doc.createElementNS("http://www.springframework.org/schema/beans", "alias");
          alias.setAttribute("name", newId);
          alias.setAttribute("alias", oldId);
          doc.getDocumentElement().appendChild(alias);
          Misc.log(Level.WARNING,
              "Replacing camel context id '%s' by '%s'. Add an alias definition for reference to old id.",
              oldId,
              newId);
        }
        el.setAttribute("id", newId);
      }
    }
    return XmlUtil.toString(doc).getBytes();
  }

  /**
   * Deploy the bundle (as input stream) to the OSGi platform.
   * Listens for application context events.
   * Throws an exception is the application context creation fails.
   *
   * @param bundleStream bundle (as stream)
   * @param bpelProcess
   * @return the application context created by this bundle
   */
  public ApplicationContext deployBundle(final InputStream bundleStream, final BpelProcess bpelProcess) {
    try {
      // Install bundle in OSGi platform
      final Bundle installedBundle =
        this.getBundleContext().installBundle("OrchestraProcess" + bpelProcess.getUUID(), bundleStream);
      // Add bundle to local map
      this.bundles.put(bpelProcess.getQName(), installedBundle);

      // Create Spring Application context listener to listen to bundle application context creation result
      final BpelBundleApplicationContextListener bundleResultListener = new BpelBundleApplicationContextListener(installedBundle);
      // register listener as a service
      final ServiceRegistration sReg =
        this.getBundleContext().registerService(
            OsgiBundleApplicationContextListener.class.getName(),
            bundleResultListener, null);
      // Start bundle
      try {
        installedBundle.start();
      } finally {
        // unregister listener
        sReg.unregister();
      }
      // Check if listener has caught an exception
      if (bundleResultListener.getFailure() != null) {
        if (bundleResultListener.getFailure() instanceof RuntimeException) {
          throw (RuntimeException) bundleResultListener.getFailure();
        } else {
          throw new OrchestraRuntimeException(bundleResultListener.getFailure());
        }
      }
      // return installed bundleapplication context
      return bundleResultListener.getApplicationContext();
    } catch (final BundleException e) {
      throw new OrchestraRuntimeException(e);
    }
  }


  @Override
  public void unpublishServices(final BpelProcess bpelProcess, final Environment environment) {
    final CxfDeployer cxfDeployer = new CxfDeployer(bpelProcess, this);
    final List<Service> services = this.getServices(bpelProcess);
    if (services != null) {
      cxfDeployer.undeploy(services);
    }
    this.deleteCamelContext(bpelProcess);
    super.unpublishServices(bpelProcess, environment);
  }

  protected void doDeploy(final BpelProcess bpelProcess) {
    try {
      this.createCamelContext(bpelProcess);

      final CxfDeployer cxfDeployer = new CxfDeployer(bpelProcess, this);
      final List<Service> services = this.getServices(bpelProcess);
      if (services != null) {
        cxfDeployer.deploy(services);
      }
    } catch (final RuntimeException e) {
      this.deleteCamelContext(bpelProcess);
      throw e;
    }
  }

  public Servlet getPublisherServlet(final Properties orchestraProperties) {
    final String context = "http://" + orchestraProperties.getProperty(OrchestraConstants.SERVLET_HOST)
        + ":" + orchestraProperties.getProperty(OrchestraConstants.SERVLET_PORT)
        + "/" + orchestraProperties.getProperty(OrchestraConstants.SERVLET_PATH);
    return new OrchestraCxfServlet(this, context);
  }

  /**
   * @return the cxfBus
   */
  public Bus getCxfBus() {
    return this.cxfBus;
  }

  /**
   * @param cxfBus the cxfBus to set
   */
  public void setCxfBus(final Bus cxfBus) {
    this.cxfBus = cxfBus;
  }

}
