/**
 * Copyright (C) 2009  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.cxf;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.wsdl.Operation;
import javax.wsdl.PortType;
import javax.xml.namespace.QName;
import javax.xml.transform.dom.DOMSource;

import org.apache.cxf.binding.soap.model.SoapBindingInfo;
import org.apache.cxf.common.util.Base64Utility;
import org.apache.cxf.message.Attachment;
import org.apache.cxf.message.Exchange;
import org.apache.cxf.message.Message;
import org.apache.cxf.service.model.BindingOperationInfo;
import org.apache.cxf.service.model.OperationInfo;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.runtime.BpelExecution;
import org.ow2.orchestra.services.MessageCarrierImpl;
import org.ow2.orchestra.services.OperationKey;
import org.ow2.orchestra.services.ReceivingService;
import org.ow2.orchestra.util.XmlUtil;
import org.ow2.orchestra.var.MessageVariable;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * Orchestra cxf web service common implementation.
 *
 * @author Guillaume Porcher
 *
 */
public final class CxfWSImpl implements org.apache.cxf.service.invoker.Invoker {

  private static Logger log = Logger.getLogger(CxfWSImpl.class.getName());

  private final PortType portType;
  private final QName processQName;
  private final ClassLoader classLoader;
  private final Map<String, Boolean> locks;
  private final CommandService commandService;

  public CxfWSImpl(
      final PortType portType,
      final QName processQName,
      final ClassLoader cl,
      final Map<String, Boolean> locks,
      final CommandService commandService) {
    this.processQName = processQName;
    this.portType = portType;
    this.classLoader = cl;
    this.locks = locks;
    this.commandService = commandService;
  }

  /**
   * Method invoked by CXF when a message arrives.
   */
  public DOMSource[] invoke(final Exchange exchange, final Object payload) {

    final ClassLoader oldClassloader = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(this.classLoader);

    try {
      /*
       * Get operation information
       */
      final BindingOperationInfo boi = exchange.get(BindingOperationInfo.class);
      final OperationInfo operationInfo = boi.getOperationInfo();

      final SoapBindingInfo bindingInfo = (SoapBindingInfo) boi.getBinding();
      // get operation style for operation
      String operationStyle = bindingInfo.getStyle(operationInfo);
      if (operationStyle == null) {
        // get operation style for service
        operationStyle = bindingInfo.getStyle();
      }
      final String operationName = operationInfo.getName().getLocalPart();

      final Operation operation = CxfWSImpl.getOperation(this.portType, operationName);

      /*
       * Process incoming message
       */
      final Message inMessage = exchange.getInMessage();


  //    if (Boolean.TRUE.equals(inMessage.get(Message.MTOM_ENABLED))) {
  //      this.processAttachments(inMessage);
  //    }

      final List<DOMSource> msgContent = inMessage.getContent(List.class);
      if (CxfWSImpl.log.isLoggable(Level.FINE)) {
        CxfWSImpl.log.fine("Entering WS Implementation, process = "
          + this.processQName + ", PT = " + this.portType.getQName() + ", op = " + operationName
          + ", opStyle = " + operationStyle);
        for (final DOMSource obj : msgContent) {
          final Document doc = (Document) obj.getNode();
          CxfWSImpl.log.fine("body = " + XmlUtil.toString(doc));
        }
      }

      // Create Orchestra inMessage from cxf message
      final javax.wsdl.Message inputMessage = operation.getInput().getMessage();
      final MessageVariable incomingMessage =
        CxfMessageUtil.cxfToOrchestraMessage(msgContent.toArray(new DOMSource[msgContent.size()]), operationStyle, inputMessage);

      if (CxfWSImpl.log.isLoggable(Level.FINE)) {
        CxfWSImpl.log.fine("incomingMessage = " + incomingMessage);
      }


      /*
       * Create message carrier if needed
       */
      MessageCarrierImpl messageCarrier = null;

      if (operation.getOutput() != null) {
        // it is a 2 ways operation. We have to get an identifier.
        messageCarrier = new MessageCarrierImpl();
      }
      if (CxfWSImpl.log.isLoggable(Level.FINE)) {
        CxfWSImpl.log.fine("Doing call to receiver, messageCarrier = " + messageCarrier);
      }

      final OperationKey operationKey = new OperationKey(this.processQName, this.portType.getQName(), operationName);

      /*
       * Call orchestra
       */
      final BpelExecution instance = ReceivingService.handle(
          incomingMessage,
          operationKey,
          messageCarrier,
          this.locks.get(operationName),
          this.commandService);
      if (CxfWSImpl.log.isLoggable(Level.FINE)) {
        CxfWSImpl.log.fine("call to receiver done. Instance = " + instance);
      }
      /*
       * Wait and process reply
       */
      if (operation.getOutput() != null) {
        if (CxfWSImpl.log.isLoggable(Level.FINE)) {
          CxfWSImpl.log.fine("This operation has an output, waiting for messageCarrier...");
        }
        final MessageVariable responseMessage = messageCarrier.getMessage();
        final QName faultQName = messageCarrier.getFaultQName();
        if (faultQName != null) {
          // throw a soap fault
          final javax.wsdl.Fault faultDef = operation.getFault(faultQName.getLocalPart());
          throw CxfMessageUtil.orchestraToCxfFault(faultQName, faultDef, responseMessage);
        }
        // Orchestra reply a message
        return CxfMessageUtil.orchestraToCxfMessage(responseMessage, operationStyle, operation.getOutput().getMessage());
      }
      // no output: return null
      return null;
    } finally {
      Thread.currentThread().setContextClassLoader(oldClassloader);
    }
  }

  /**
   * Process inMessage. Gets all the XOP includes and inlines attachements back into the xml document.
   * @param inMessage
   */
  private void processAttachments(final Message inMessage) {
    final List<DOMSource> msgContent = inMessage.getContent(List.class);
    for (final DOMSource obj : msgContent) {
      final Document doc = (Document) obj.getNode();
      // Process XOP includes
      // get nodes
      final NodeList includeNodes = doc.getElementsByTagNameNS("http://www.w3.org/2004/08/xop/include", "Include");
      for (int i = 0; i < includeNodes.getLength(); i++) {
        final Element n = (Element) includeNodes.item(i);
        // get href:
        String attachmentId = XmlUtil.attribute(n, "href");
        if (attachmentId.startsWith("cid:")) {
          attachmentId = attachmentId.substring(4);
        }
        // get attachment
        for (final Attachment attachment : inMessage.getAttachments()) {
          if (attachment.getId().equals(attachmentId)) {
            final ByteArrayOutputStream baos = new ByteArrayOutputStream();
            InputStream inStream;
            try {
              inStream = attachment.getDataHandler().getInputStream();
              final byte[] buffer = new byte[1024];
              int c;
              while ((c = inStream.read(buffer)) != -1) {
                baos.write(buffer, 0, c);
              }
            } catch (final IOException e) {
              throw new OrchestraRuntimeException(e);
            }
            n.getParentNode().replaceChild(doc.createTextNode(Base64Utility.encode(baos.toByteArray())), n);
            break;
          }
        }
      }
    }
  }

  private static Operation getOperation(final PortType portType, final String operationName) {
    final List<Operation> operations = portType.getOperations();
    for (final Operation operation : operations) {
      if (operation.getName().equals(operationName)) {
        return operation;
      }
    }
    return null;
  }

}
