/**
 * Copyright (C) 2009  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.cxf;

import java.util.ArrayList;
import java.util.List;

import javax.wsdl.Part;
import javax.xml.namespace.QName;
import javax.xml.transform.dom.DOMSource;

import org.apache.cxf.binding.soap.SoapFault;
import org.apache.cxf.interceptor.Fault;
import org.ow2.orchestra.util.BpelSOAPUtil;
import org.ow2.orchestra.util.XmlUtil;
import org.ow2.orchestra.var.MessageVariable;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * @author Guillaume Porcher
 *
 */
public final class CxfMessageUtil {

  private CxfMessageUtil() { }

  public static MessageVariable cxfToOrchestraMessage(
      final DOMSource[] cxfMessage, final Object operationStyle, final javax.wsdl.Message wsdlMessage) {
    final List<Element> bodies = new ArrayList<Element>();
    for (final DOMSource obj : cxfMessage) {
      final Document doc = (Document) obj.getNode();
      bodies.add(doc.getDocumentElement());
    }

    MessageVariable incomingMessage = null;

    if (operationStyle.equals("document")) {
      incomingMessage = BpelSOAPUtil.buildMessageFromDocumentSOAPBodyElement(wsdlMessage.getOrderedParts(null), bodies.get(0));
    } else {
      incomingMessage = BpelSOAPUtil.buildMessageFromRpcSOAPBodyElement(wsdlMessage.getOrderedParts(null), bodies.get(0));
    }

    return incomingMessage;
  }

  public static DOMSource[] orchestraToCxfMessage(
      final MessageVariable orchestraMessage, final Object operationStyle, final javax.wsdl.Message wsdlMessage) {
    Element[] elementsReturned;

    if (operationStyle.equals("document")) {

      // We make the response");
      final List<Part> parts = wsdlMessage.getOrderedParts(null);
      elementsReturned = new Element[parts.size()];
      int i = 0;
      for (final Part part : parts) {
        elementsReturned[i] = orchestraMessage.getPartValue(part.getName());
        i++;
      }
    } else {
      // it is a rpc style!!
      // We make the response");
      final Document rpcDocument = BpelSOAPUtil.buildRpcDocument(orchestraMessage, "temp");
      final List<Element> rpcElements = XmlUtil.elements(rpcDocument.getDocumentElement());
      elementsReturned = rpcElements.toArray(new Element[rpcElements.size()]);
    }

    final DOMSource[] result = new DOMSource[elementsReturned.length];
    for (int i = 0; i < elementsReturned.length; i++) {
      result[i] = new DOMSource(elementsReturned[i]);
    }
    return result;
  }

  public static Fault orchestraToCxfFault(
      final QName faultQName, final javax.wsdl.Fault faultDef, final MessageVariable orchestraFaultMessage) {
    // throw fault
    final Fault fault = new SoapFault("cxf fault", faultQName);
    final Element detail = fault.getOrCreateDetail();

    // We make the fault detail;
    if (faultDef != null) {
      // WSDL declared fault
      final javax.wsdl.Message faultMessage = faultDef.getMessage();
      final List<Part> parts = faultMessage.getOrderedParts(null);
      for (final Part part : parts) {
        final Node importedNode = detail.getOwnerDocument().importNode(orchestraFaultMessage.getPartValue(part.getName()), true);
        detail.appendChild(importedNode);
      }

    } else {
      if (orchestraFaultMessage.getParts() != null && orchestraFaultMessage.getParts().size() == 1) {
        final Node importedNode = detail.getOwnerDocument().importNode(
            orchestraFaultMessage.getParts().values().iterator().next(), true);
        detail.appendChild(importedNode);
      }
    }
    return fault;
  }
}
