package org.ow2.orchestra.cxf;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.wsdl.Definition;
import javax.wsdl.OperationType;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.wsdl.extensions.soap.SOAPAddress;
import javax.wsdl.extensions.soap.SOAPBinding;
import javax.wsdl.extensions.soap.SOAPBody;
import javax.wsdl.extensions.soap.SOAPFault;
import javax.wsdl.extensions.soap12.SOAP12Address;
import javax.wsdl.extensions.soap12.SOAP12Binding;
import javax.wsdl.extensions.soap12.SOAP12Body;
import javax.wsdl.extensions.soap12.SOAP12Fault;
import javax.xml.namespace.QName;

import org.apache.camel.CamelContext;
import org.apache.camel.component.cxf.transport.CamelDestination;
import org.apache.camel.component.cxf.transport.CamelTransportFactory;
import org.apache.cxf.endpoint.Server;
import org.apache.cxf.endpoint.ServerRegistry;
import org.apache.cxf.frontend.ServerFactoryBean;
import org.apache.cxf.interceptor.OneWayProcessorInterceptor;
import org.apache.cxf.service.factory.ReflectionServiceFactoryBean;
import org.apache.cxf.service.model.EndpointInfo;
import org.apache.cxf.service.model.OperationInfo;
import org.apache.cxf.wsdl11.WSDLServiceFactory;
import org.ow2.orchestra.definition.BpelProcess;
import org.ow2.orchestra.env.Environment;
import org.ow2.orchestra.services.commands.CommandService;
import org.ow2.orchestra.ws.WSDeployer;

/**
 * CXF engine WS deployer.
 *
 * @author Guillaume Porcher
 */
public class CxfDeployer extends WSDeployer {

  private final CxfPublisher cxfPublisher;
  /**
   * Default constructor.
   *
   * @param wsId - id of the WS : in our case, processName
   * @param wsdlUrl - url of the wsdl file to deploy.
   * @param orchestraDirectory - Absolute path to the orchestra directory.
   */
  public CxfDeployer(final BpelProcess bpelProcess, final CxfPublisher cxfPublisher) {
    super(bpelProcess);
    this.cxfPublisher = cxfPublisher;
  }

  /**
   * return null if operation style is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param operationStyle - operationStyle to check
   * @return null if operation style is supported by Axis engine. Else, returns
   *         a message explaining the problem.
   */
  @Override
  protected String checkOperationStyle(final String operationStyle) {
    if (operationStyle == null) {
      return "Style attribute of this operation must be specified";
    } else if (!operationStyle.equals("document") && !operationStyle.equals("rpc")) {
      return "Style attribute of this operation must be : document or rpc";
    }
    return null;
  }

  /**
   * return null if operation type is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param operationType - operationType to check
   * @return null if operation type is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  @Override
  protected String checkOperationType(final OperationType operationType) {
    if (!operationType.equals(OperationType.REQUEST_RESPONSE) && !operationType.equals(OperationType.ONE_WAY)) {
      return "Operation type : "
        + operationType + " is not supported. Please use one of : "
        + OperationType.ONE_WAY + "/" + OperationType.REQUEST_RESPONSE;
    }
    return null;
  }

  /**
   * return null if soapVersion is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param soapBinding - soapBinding to check
   * @return null if soapVersion is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  @Override
  protected String checkSoapVersion(final ExtensibilityElement soapBinding) {
    if (!(soapBinding instanceof SOAPBinding) && !(soapBinding instanceof SOAP12Binding)) {
      return "Supported Soap Version are " + WSDeployer.URI_WSDL11_SOAP11 + "/" + WSDeployer.URI_WSDL11_SOAP12;
    }
    return null;
  }

  /**
   * return null if transport is supported by Axis engine. Else, returns a
   * message explaining the problem.
   *
   * @param soapBinding - soapBinding to check
   * @return null if transport is supported by Axis engine. Else, returns a
   *         message explaining the problem.
   */
  @Override
  protected String checkTransport(final ExtensibilityElement soapBinding) {
    String transportUri = "";
    if (soapBinding instanceof SOAPBinding) {
      transportUri = ((SOAPBinding) soapBinding).getTransportURI();
    } else if (soapBinding instanceof SOAP12Binding) {
      transportUri = ((SOAP12Binding) soapBinding).getTransportURI();
    }
    if (WSDeployer.SOAP_HTTP_TRANSPORT_URI.equals(transportUri)) {
      return null;
    }
    if (CamelTransportFactory.TRANSPORT_ID.endsWith(transportUri)) {
      return null;
    }
    return "Transport URI : " + transportUri + " is not supported. Please use "
      + WSDeployer.SOAP_HTTP_TRANSPORT_URI + " or " + CamelTransportFactory.TRANSPORT_ID;
  }

  /**
   * return null if use is supported by Axis engine. Else, returns a message
   * explaining the problem.
   *
   * @param soapBody - soapBody to check
   * @return null if use is supported by Axis engine. Else, returns a message
   *         explaining the problem.
   */
  @Override
  protected String checkUse(final ExtensibilityElement element) {
    String use = "";
    if (element instanceof SOAPBody) {
      use = ((SOAPBody) element).getUse();
    } else if (element instanceof SOAP12Body) {
      use = ((SOAP12Body) element).getUse();
    } else if (element instanceof SOAPFault) {
      use = ((SOAPFault) element).getUse();
    } else if (element instanceof SOAP12Fault) {
      use = ((SOAP12Fault) element).getUse();
    }
    if (!"literal".equals(use)) {
      return "Use : " + use + " is not supported. Please use " + "literal";
    }
    return null;
  }

  /**
   * return null if soapBody attributes are supported by Axis engine. Else,
   * returns a message explaining the problem.
   *
   * @param soapBody - soapBody to check
   * @return null if soapBody attributes are supported supported by Axis engine.
   *         Else, returns a message explaining the problem.
   */
  @Override
  protected String checkSoapBody(final ExtensibilityElement soapBody) {
    List parts = null;
    if (soapBody instanceof SOAPBody) {
      parts = ((SOAPBody) soapBody).getParts();
    } else if (soapBody instanceof SOAP12Body) {
      parts = ((SOAP12Body) soapBody).getParts();
    }
    if (parts != null) {
      return "SoapBody is using parts attribute which is not currently supported.";
    }
    return null;
  }

  @Override
  protected String checkSoapFault(final ExtensibilityElement soapFault) {
    String name = null;
    if (soapFault instanceof SOAPFault) {
      name = ((SOAPFault) soapFault).getName();
    } else if (soapFault instanceof SOAP12Fault) {
      name = ((SOAP12Fault) soapFault).getName();
    }
    if (name == null) {
      return "SoapFault is not specifying fault name which is not currently supported.";
    }
    return null;
  }



  /**
   * This method will deploy the specified WS Service on the choosen ws engine.
   * This method must be overriden by each ws engine Deployer.
   *
   * @param def - Definition object that represents a WS Service to deploy.
   */
  @SuppressWarnings("unchecked")
  @Override
  protected void deployServices(final List<Service> services) {
    for (final Service service : services) {

      final QName serviceQName = service.getQName();

      final Map<String, Port> ports = service.getPorts();
      for (final Port port : ports.values()) {

        final QName endpointQName = new QName(serviceQName.getNamespaceURI(), port.getName());

        final ServerFactoryBean svrBean = new ServerFactoryBean();

        // set cxf bus
        svrBean.setBus(this.cxfPublisher.getCxfBus());

        // set address of the WS
        final String endpointURL = this.getAddressFromPort(port);
        if (endpointURL != null) {
          svrBean.setAddress(endpointURL);
        } else {
          svrBean.setAddress("/" + port.getName());
        }

        // set service name
        svrBean.setServiceName(serviceQName);

        // get WSDL definition of the service
        final Definition wsdlDefinition = this.bpelProcess.getWsdlInfos().extractServiceWsdlDefinition(service);
        // Create a service factory from the definition
        final WSDLServiceFactory wsdlServiceFactory =
          new WSDLServiceFactory(this.cxfPublisher.getCxfBus(), wsdlDefinition, serviceQName);
        wsdlServiceFactory.setEndpointName(endpointQName);

        final org.apache.cxf.service.Service cxfService = wsdlServiceFactory.create();

        final EndpointInfo endpointInfo = cxfService.getEndpointInfo(endpointQName);
        // unset wrapped attribute in operation: we want cxf to return the elements as defined in the wsdl
        for (final OperationInfo opInfo : endpointInfo.getInterface().getOperations()) {
          opInfo.setUnwrappedOperation(null);
        }
        // execute one way operations in the same thread
        endpointInfo.setProperty(OneWayProcessorInterceptor.USE_ORIGINAL_THREAD, Boolean.TRUE);

        final QName portTypeQname = port.getBinding().getPortType().getQName();
        final ReflectionServiceFactoryBean rsfb =
          new OrchestraReflectionServiceFactoryBean(portTypeQname, cxfService);
        rsfb.setServiceName(serviceQName);
        rsfb.setEndpointName(endpointQName);

        final Map<String, Boolean> locks = new HashMap<String, Boolean>();
        final Map<String, Boolean> isOneWay = new HashMap<String, Boolean>();

        this.getLocksAndOneWays(port, locks, isOneWay);

        rsfb.setInvoker(
            new CxfWSImpl(
                port.getBinding().getPortType(),
                this.bpelProcess.getQName(),
                Environment.getCurrent().getClassLoader(),
                locks,
                Environment.getFromCurrent(CommandService.class)
          )
        );

        svrBean.setServiceFactory(rsfb);

        // create server
        svrBean.setStart(false);
        final Server server = svrBean.create();

        if (server.getDestination() instanceof CamelDestination) {
          final CamelDestination camelDestination = (CamelDestination) server.getDestination();
          final CamelContext camelContext = this.cxfPublisher.getCamelContext(this.bpelProcess.getQName());
          if (camelContext != null) {
            camelDestination.setCamelContext(camelContext);
          }
        }
        // setup interceptors
        CxfUtils.setUpOrchestraInterceptors(server.getEndpoint());

        // register process endpoint if it is not a camel endpoint
        // process can be called from cxf or directly from another process
        if (endpointURL == null
            || !endpointURL.startsWith(this.expectedAddressPrefix)
            || this.cxfPublisher.addProcessEndpoint(
                endpointURL, this.bpelProcess.getQName(), port, this.bpelProcess.getUUID(), locks, isOneWay)) {
          // start server only when camel endpoint or new process endpoint.
          server.start();
        }
        if (endpointURL != null) {
          // reset the endpoint for a port
          final List<ExtensibilityElement> extensibilityList = port.getExtensibilityElements();
          for (final ExtensibilityElement obj : extensibilityList) {
            if (obj instanceof SOAPAddress) {
              ((SOAPAddress) obj).setLocationURI(endpointURL);
            } else if (obj instanceof SOAP12Address) {
              ((SOAP12Address) obj).setLocationURI(endpointURL);
            }
          }
        }
      }
    }
  }

  @Override
  protected boolean isAddressSupported(final String address) {
    return address.startsWith("camel:") || super.isAddressSupported(address);
  }

  /**
   * This method will undeploy the specified WS Service from the choosen ws
   * engine. This method must be overriden by each ws engine Deployer.
   *
   * @param def - Definition object that represents a WS Service to undeploy.
   */
  @Override
  protected void undeployServices(final List<Service> services) {
    final ServerRegistry serverRegistry = this.cxfPublisher.getCxfBus().getExtension(ServerRegistry.class);

    for (final Service service : services) {
      final Map<String, Port> ports = service.getPorts();
      for (final Port port : ports.values()) {
        // set address of the WS
        final String endpointURL = this.getAddressFromPort(port);
        final QName endpointQName = new QName(service.getQName().getNamespaceURI(), port.getName());
        final List<Server> servers = serverRegistry.getServers();
        if (servers != null) {
          for (final Server server : new ArrayList<Server>(servers)) {
            if (server.getEndpoint().getService().getName().equals(service.getQName())
                && server.getEndpoint().getEndpointInfo().getName().equals(endpointQName)) {
              // found server matching the endpoint QName
              // verify the server endpoint is the endPoint URL
              final String serverEndpointURL =
                  // remove tailing / from addressPrefix
                  (this.expectedAddressPrefix.endsWith("/")
                      ? this.expectedAddressPrefix.substring(0, this.expectedAddressPrefix.lastIndexOf('/'))
                      : this.expectedAddressPrefix)
                  // append server endpoint info
                  + server.getEndpoint().getEndpointInfo().getAddress();
              // stop service only if no other versions of the process use it
              // or if this is not a web service endpoint (camel endpoint).
              if (!endpointURL.startsWith(this.expectedAddressPrefix)
                  || (endpointURL.equals(serverEndpointURL)
                      && this.cxfPublisher.removeProcessEndpoint(endpointURL, this.bpelProcess.getUUID()))) {
                server.stop();
              }
            }
          }
        }
      }
    }
  }

  /* (non-Javadoc)
   * @see org.ow2.orchestra.ws.WSDeployer#checkWSEngineIsAvailable()
   */
  @Override
  protected void checkWSEngineIsAvailable() {
    // TODO
  }


}
