/**
 * Copyright (C) 2009  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.cxf;

import java.util.Collection;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import javax.wsdl.Definition;
import javax.wsdl.Fault;
import javax.wsdl.Operation;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.xml.namespace.QName;
import javax.xml.transform.dom.DOMSource;

import org.apache.camel.CamelContext;
import org.apache.camel.component.cxf.transport.CamelConduit;
import org.apache.cxf.binding.soap.SoapFault;
import org.apache.cxf.binding.soap.model.SoapBindingInfo;
import org.apache.cxf.endpoint.Client;
import org.apache.cxf.frontend.ClientFactoryBean;
import org.apache.cxf.service.factory.ReflectionServiceFactoryBean;
import org.apache.cxf.service.model.BindingOperationInfo;
import org.apache.cxf.service.model.OperationInfo;
import org.apache.cxf.wsdl11.WSDLServiceFactory;
import org.ow2.orchestra.env.Environment;
import org.ow2.orchestra.exception.FaultWithMessageVariable;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.services.OperationKey;
import org.ow2.orchestra.services.itf.Invoker;
import org.ow2.orchestra.util.AddressingUtil.AddressingInfo;
import org.ow2.orchestra.util.BpelSOAPUtil;
import org.ow2.orchestra.util.XmlUtil;
import org.ow2.orchestra.util.wsdl.WsdlUtil;
import org.ow2.orchestra.var.MessageVariable;
import org.w3c.dom.Element;


/**
 * @author Guillaume Porcher
 *
 */
public class CxfInvoker implements Invoker {

  private final CxfPublisher cxfPublisher;

  public CxfInvoker() {
    this.cxfPublisher = Environment.getFromCurrent(CxfPublisher.class);
  }

  public MessageVariable invoke(final OperationKey operationKey,
      final AddressingInfo addressingInfo,
      final MessageVariable requestMessage,
      final Set<Definition> wsdlDefinitions,
      final Properties orchestraProperties) {
    /*
     * Get port and service from addressing properties
     */
    Definition definition = addressingInfo.getWsdlDefinition();
    Service service = addressingInfo.getService();
    Port port = addressingInfo.getPort();

    if (port == null) {
      /*
       * find port to call in wsdl definition
       */
      final List<Service> services = WsdlUtil.getServicesOfPortType(operationKey.getPortTypeQName(), wsdlDefinitions);
      addressingInfo.resolveServiceAndPort(services);

      port = addressingInfo.getPort();
      service = addressingInfo.getService();

      /*
       * If no service found from addressing informations, use first available service.
       */
      if (service == null) {
        service = services.get(0);
        for (final Port p : (Collection<Port>) service.getPorts().values()) {
          if (p.getBinding().getPortType().getQName().equals(operationKey.getPortTypeQName())) {
            port = p;
            break;
          }
        }
      }
      /*
       * Find wsdl definition of the service
       */
      for (final Definition wsdlDefinition : wsdlDefinitions) {

        if (service.equals(wsdlDefinition.getService(service.getQName()))) {
          definition = wsdlDefinition;
          break;
        }
      }
    }

    if (service == null || definition == null || port == null) {
      throw new OrchestraRuntimeException("Can't find any supported service!");
    }

    /*
     * Create a CXF client for this port.
     */
    final Client client = this.createCxfClient(operationKey, addressingInfo, definition, service, port);

    /*
     * get operation style for operation to invoke
     */
    final BindingOperationInfo boi = client.getEndpoint().getEndpointInfo().getBinding().getOperation(
        new QName(operationKey.getPortTypeQName().getNamespaceURI(), operationKey.getOperationName()));
    final OperationInfo operationInfo = boi.getOperationInfo();
    final SoapBindingInfo bindingInfo = (SoapBindingInfo) boi.getBinding();
    // get operation style for operation
    String operationStyle = bindingInfo.getStyle(operationInfo);
    if (operationStyle == null) {
      // get operation style for service
      operationStyle = bindingInfo.getStyle();
    }

    final Operation operation = port.getBinding().getPortType().getOperation(operationKey.getOperationName(), null, null);

    try {
      /*
       * prepare cxf message
       */
      final DOMSource[] outMessage =
        CxfMessageUtil.orchestraToCxfMessage(
            requestMessage, operationStyle, operation.getInput().getMessage());
      /*
       * Invoke web service
       */
      final Object[] res = client.invoke(operationKey.getOperationName(), (Object[]) outMessage);

      if (operation.getOutput() != null) {
        /*
         * process response
         */
        final DOMSource[] domRes = new DOMSource[res.length];
        for (int i = 0; i < res.length; i++) {
          domRes[i] = (DOMSource) res[i];
        }
        return CxfMessageUtil.cxfToOrchestraMessage(domRes, operationStyle, operation.getOutput().getMessage());
      } else {
        // operation.getOutput() is null
        return null;
      }
    } catch (final SoapFault s) {
      /*
       * Handle fault returned by web service
       */
      final QName faultName = s.getFaultCode();
      MessageVariable faultMessage = null;
      if (operationKey.getPortTypeQName().getNamespaceURI().equals(faultName.getNamespaceURI())) {
        // This is a WSDL fault, get message
        final Fault fault = operation.getFault(faultName.getLocalPart());
        if (fault != null && s.hasDetails()) {
          final Element detailEntry = s.getDetail();
          // style for faults is always document (cf http://www.w3.org/TR/wsdl#_soap:fault )
          faultMessage = BpelSOAPUtil.buildMessageFromDocumentSOAPBodyElement(
              fault.getMessage().getOrderedParts(null), XmlUtil.element(detailEntry));
        }
      }
      throw new FaultWithMessageVariable(faultName, faultMessage);
    } catch (final Exception e) {
      if (e instanceof OrchestraRuntimeException) {
        throw (OrchestraRuntimeException) e;
      }
      throw new OrchestraRuntimeException(e);
    }
  }

  private Client createCxfClient(final OperationKey operationKey,
      final AddressingInfo addressingInfo, final Definition definition, final Service service, final Port port) {
    // endpoint is the port
    final QName endpointQName = new QName(service.getQName().getNamespaceURI(), port.getName());
    // create a cxf service from wsdl definition
    final WSDLServiceFactory wsdlServiceFactory =
      new WSDLServiceFactory(this.cxfPublisher.getCxfBus(), definition, service.getQName());
    wsdlServiceFactory.setEndpointName(endpointQName);

    final org.apache.cxf.service.Service cxfService = wsdlServiceFactory.create();

    // unset wrapped attribute in operation: we want cxf to return the elements as defined in the wsdl
    for (final OperationInfo opInfo : cxfService.getEndpointInfo(endpointQName).getInterface().getOperations()) {
      opInfo.setUnwrappedOperation(null);
    }

    final ReflectionServiceFactoryBean rsfb =
      new OrchestraReflectionServiceFactoryBean(operationKey.getPortTypeQName(), cxfService);
    rsfb.setServiceName(service.getQName());
    rsfb.setEndpointName(endpointQName);

    // create cxf client
    final ClientFactoryBean cliBean = new ClientFactoryBean(rsfb);

    cliBean.setAddress(addressingInfo.getAddress());

    final Client client = cliBean.create();

    if (client.getConduit() instanceof CamelConduit) {
      final CamelContext camelContext = this.cxfPublisher.getCamelContext(operationKey.getProcessQName());
      final CamelConduit camelConduit = (CamelConduit) client.getConduit();
      if (camelContext != null) {
        camelConduit.setCamelContext(camelContext);
      }
    }
    // setup interceptors
    CxfUtils.setUpOrchestraInterceptors(client.getEndpoint());
    return client;
  }

}
