/**
 * Copyright (C) 2009  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.cxf;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.camel.component.cxf.transport.CamelTransportFactory;
import org.apache.cxf.Bus;
import org.apache.cxf.management.InstrumentationManager;
import org.apache.cxf.management.jmx.InstrumentationManagerImpl;
import org.apache.cxf.transport.ConduitInitiatorManager;
import org.apache.cxf.transport.DestinationFactory;
import org.apache.cxf.transport.DestinationFactoryManager;
import org.apache.cxf.transport.servlet.CXFServlet;

public class OrchestraCxfServlet extends CXFServlet {

  private static final long serialVersionUID = 2528286722832028480L;

  private final CxfPublisher cxfPublisher;
  private final String context;


  public OrchestraCxfServlet(final CxfPublisher cxfPublisher, final String context) {
    this.cxfPublisher = cxfPublisher;
    this.context = context;
  }

  /* (non-Javadoc)
   * @see org.apache.cxf.transport.servlet.CXFServlet#loadBus(javax.servlet.ServletConfig)
   */
  @Override
  public void loadBus(final ServletConfig servletConfig) throws ServletException {

    // Load bus in cxf class loader to find cxf configuration files.
    final ClassLoader oldCl = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(CXFServlet.class.getClassLoader());
    try {
      super.loadBus(servletConfig);
    } finally {
      Thread.currentThread().setContextClassLoader(oldCl);
    }

    // Register camel transport
    final CamelTransportFactory camelTransportFactory = new CamelTransportFactory();
    // register the conduit initiator
    final ConduitInitiatorManager cim = this.bus.getExtension(ConduitInitiatorManager.class);
    cim.registerConduitInitiator(CamelTransportFactory.TRANSPORT_ID, camelTransportFactory);
    // register the destination factory
    final DestinationFactoryManager dfm = this.bus.getExtension(DestinationFactoryManager.class);
    dfm.registerDestinationFactory(CamelTransportFactory.TRANSPORT_ID, camelTransportFactory);

    // register mbeans
    final InstrumentationManager instrumentationManager = this.bus.getExtension(InstrumentationManager.class);
    if (instrumentationManager != null && instrumentationManager instanceof InstrumentationManagerImpl) {
      final InstrumentationManagerImpl instrumentationManagerImpl = (InstrumentationManagerImpl) instrumentationManager;
      instrumentationManagerImpl.setUsePlatformMBeanServer(true);
// Disable mbeans: mbeans are not unregistered when endpoints are stopped. (https://issues.apache.org/jira/browse/CXF-2740)
//      instrumentationManagerImpl.setEnabled(true);
      instrumentationManagerImpl.setBus(this.bus);
      instrumentationManagerImpl.setCreateMBServerConnectorFactory(false);
      instrumentationManagerImpl.init();
    }

    // set or bus as the bus for cxf publisher
    this.cxfPublisher.setCxfBus(this.bus);
  }

  @Override
  protected DestinationFactory createServletTransportFactory() {
    if (this.servletTransportFactory == null) {
      this.servletTransportFactory = new OrchestraCxfServletTransportFactory(this.bus, this.context);
    }
    return this.servletTransportFactory;
  }

  @Override
  protected void invoke(final HttpServletRequest request, final HttpServletResponse response) throws ServletException {
    final ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
    try {
      Thread.currentThread().setContextClassLoader(Bus.class.getClassLoader());
      super.invoke(request, response);
    } finally {
      Thread.currentThread().setContextClassLoader(oldClassLoader);
    }
  }
}
