/**
 * Copyright (C) 2009  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.cxf;

import java.util.Properties;

import org.apache.camel.component.cxf.transport.CamelTransportFactory;
import org.apache.cxf.Bus;
import org.apache.cxf.endpoint.Server;
import org.apache.cxf.jaxrs.JAXRSServerFactoryBean;
import org.apache.cxf.jaxrs.provider.JAXBElementProvider;
import org.apache.cxf.management.InstrumentationManager;
import org.apache.cxf.management.jmx.InstrumentationManagerImpl;
import org.apache.cxf.transport.ConduitInitiatorManager;
import org.apache.cxf.transport.DestinationFactoryManager;
import org.apache.cxf.transport.servlet.CXFServlet;
import org.ow2.orchestra.env.EnvironmentFactory;
import org.ow2.orchestra.rest.JaxRSInstanceManagementAPIImpl;
import org.ow2.orchestra.rest.JaxRSManagementAPIImpl;
import org.ow2.orchestra.rest.JaxRSQueryDefinitionAPIImpl;
import org.ow2.orchestra.rest.JaxRSQueryRuntimeAPIImpl;
import org.ow2.orchestra.util.JAXBUtils;
import org.ow2.orchestra.util.OrchestraConstants;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

public class OrchestraCxfServlet extends CXFServlet {

  private static final long serialVersionUID = 2528286722832028480L;

  private final CxfPublisher cxfPublisher;
  private final String context;
  private final EnvironmentFactory environmentFactory;


  public OrchestraCxfServlet(final CxfPublisher cxfPublisher, final String context, EnvironmentFactory environmentFactory) {
    this.cxfPublisher = cxfPublisher;
    this.context = context;
    this.environmentFactory = environmentFactory;
  }

  /* (non-Javadoc)
   * @see org.apache.cxf.transport.servlet.CXFServlet#loadBus(javax.servlet.ServletConfig)
   */
  @Override
  public void loadBus(final ServletConfig servletConfig) {

    // Load bus in cxf class loader to find cxf configuration files.
    final ClassLoader oldCl = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(CXFServlet.class.getClassLoader());
    try {
      super.loadBus(servletConfig);
    } finally {
      Thread.currentThread().setContextClassLoader(oldCl);
    }

    // Register camel transport
    final CamelTransportFactory camelTransportFactory = new OrchestraCamelTransportFactory(this.cxfPublisher);

    // register the conduit initiator
    final ConduitInitiatorManager cim = this.getBus().getExtension(ConduitInitiatorManager.class);
    cim.registerConduitInitiator(CamelTransportFactory.TRANSPORT_ID, camelTransportFactory);
    // register the destination factory
    final DestinationFactoryManager dfm = this.getBus().getExtension(DestinationFactoryManager.class);
    dfm.registerDestinationFactory(CamelTransportFactory.TRANSPORT_ID, camelTransportFactory);

    // register mbeans
    final InstrumentationManager instrumentationManager = this.getBus().getExtension(InstrumentationManager.class);
    if (instrumentationManager != null && instrumentationManager instanceof InstrumentationManagerImpl) {
      final InstrumentationManagerImpl instrumentationManagerImpl = (InstrumentationManagerImpl) instrumentationManager;
      instrumentationManagerImpl.setUsePlatformMBeanServer(true);
      instrumentationManagerImpl.setEnabled(true);
      instrumentationManagerImpl.setBus(this.getBus());
      instrumentationManagerImpl.setCreateMBServerConnectorFactory(false);
      instrumentationManagerImpl.init();
    }
    // Update http destination factory
    new OrchestraCxfServletTransportFactory(this.getBus(), this.context);

    // Register Rest APIs
    this.registerRESTApis();

    // set or bus as the bus for cxf publisher
    this.cxfPublisher.setCxfBus(this.getBus());
  }

  private void registerRESTApis() {
    // Register Rest APIs
    JAXRSServerFactoryBean sf = new JAXRSServerFactoryBean();
    sf.setServiceBeanObjects(
            new JaxRSManagementAPIImpl(this.environmentFactory),
            new JaxRSInstanceManagementAPIImpl(this.environmentFactory),
            new JaxRSQueryDefinitionAPIImpl(this.environmentFactory),
            new JaxRSQueryRuntimeAPIImpl(this.environmentFactory));      
    sf.setAddress("/apis");
    JAXBElementProvider provider = new JAXBElementProvider();
    provider.setCollectionWrapperName("collection");
    provider.setSingleJaxbContext(true);
    provider.setExtraClass(JAXBUtils.JAXB_CLASSES);
    sf.setProvider(provider);
    sf.setBus(this.getBus());
    sf.setStart(false);
    Server s = sf.create();
    s.start();
  }


  @Override
  protected void invoke(final HttpServletRequest request, final HttpServletResponse response) throws ServletException {
    final ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
    try {
      Thread.currentThread().setContextClassLoader(Bus.class.getClassLoader());
      super.invoke(request, response);
    } finally {
      Thread.currentThread().setContextClassLoader(oldClassLoader);
    }
  }
}
