package org.ow2.orchestra.jmxclient;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;

import javax.xml.namespace.QName;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.novabpm.util.Misc;
import org.ow2.orchestra.exception.OrchestraException;
import org.ow2.orchestra.jmx.RemoteDeployerMBean;

public class DeployBarAction implements ClientAction {

  private Options options = ClientOptions.createDeployBarOptions();

  /**
   * Deploy a BPEL process from a Bar
   *
   * @param barFile
   *            URL of the bar file to deploy
   * @param jmxServiceUrl
   *            url of the jmx service
   * @param jmxObjectName
   *            name of the remote deployer
   * @return QName of the deployed process
   */
  public static QName deployBar(URL barFile, String jmxServiceUrl,
      String jmxObjectName) {
    QName processName = null;
    byte[] barBytes = null;
    try {
      InputStream stream = barFile.openStream();
      ByteArrayOutputStream output = new ByteArrayOutputStream();
      int c;
      while ((c = stream.read()) != -1) {
        output.write(c);
      }
      stream.close();
      barBytes = output.toByteArray();
    } catch (Exception e) {
      throw new OrchestraException(
          "couldn't read bar file from url " + barFile, e);
    }
    try {
      RemoteDeployerMBean remoteDeployer = Misc.getMBeanProxy(RemoteDeployerMBean.class, jmxServiceUrl, jmxObjectName);
      processName = remoteDeployer.deployBar(barBytes);
    } catch (Exception e) {
      throw new OrchestraException("Exception while deploying barFile : "
          + barFile, e);
    }
    return processName;
  }

  /**
   * Deploy a BPEL process from a Bar
   *
   * @param barBytes
   *            bytes array of the bar file to deploy
   * @param jmxServiceUrl
   *            url of the jmx service
   * @param jmxObjectName
   *            name of the remote deployer
   * @return QName of the deployed process
   */
  public static QName deployBar(byte[] barBytes, String jmxServiceUrl,
      String jmxObjectName) {
    QName processName = null;
    try {
      RemoteDeployerMBean remoteDeployer = Misc.getMBeanProxy(RemoteDeployerMBean.class, jmxServiceUrl, jmxObjectName);
      processName = remoteDeployer.deployBar(barBytes);
    } catch (Exception e) {
      throw new OrchestraException("Exception while deploying barFile ", e);
    }
    return processName;
  }

  /**
   * Deploy a process
   *
   * @param line
   *            parsed command line
   */
  protected static void deployBar(CommandLine line) {
    URL barFile = null;
    String jmxServiceUrl = line.getOptionValue(ClientOptions.JMX_URL_OPTION,
        JMXClient.jmxServiceUrl);
    String jmxObjectName = line.getOptionValue(
        ClientOptions.JMX_OBJECT_OPTION, JMXClient.jmxObjectName);

    if (line.hasOption(ClientOptions.BAR_OPTION)) {
      String barFileName = line.getOptionValue(ClientOptions.BAR_OPTION);
      try {
        barFile = new File(barFileName).toURL();
      } catch (MalformedURLException e) {
        throw new OrchestraException("Invalid bar url " + barFileName, e);
      }
    }
    QName processName = deployBar(barFile, jmxServiceUrl, jmxObjectName);
    System.out.println("Process " + processName + " deployed !");

  }

  public boolean parseAndExecute(String[] args) {
    // create the parser
    CommandLineParser parser = new GnuParser();
    CommandLine line = null;
    try {
      // parse the command line arguments
      line = parser.parse(options, args);
    } catch (ParseException exp) {
      // oops, something went wrong
      return false;
    }

    deployBar(line);

    return true;
  }

  public Options getOptions() {
    return options;
  }

  public String getDescription() {
    return "To deploy a BPEL process from a Bar";
  }

}
