package org.ow2.orchestra.jmxclient;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.novabpm.util.Misc;
import org.ow2.orchestra.facade.Deployment;
import org.ow2.orchestra.facade.def.ProcessDefinition;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.facade.jmx.RemoteDeployerMBean;
import org.ow2.orchestra.parsing.BpelImportParser;

public class DeployAction implements ClientAction {

  private Options options = ClientOptions.createDeployOptions();

  /**
   * Deploy a BPEL process
   *
   * @param bpelFile
   *            URL of the bpel file to deploy
   * @param wsdlFile
   *            URL of the wsdl file
   * @param envFile
   *            URL of the environment xml
   * @param jmxServiceUrl
   *            url of the jmx service
   * @param jmxObjectName
   *            name of the remote deployer
   * @return QName of the deployed process
   */
  public static ProcessDefinition deploy(final URL bpelURL, final List<URL> wsdls,
      final String jmxServiceUrl, final String jmxObjectName) {
    ProcessDefinition processName = null;
    try {
      Deployment bpelDeployment = BpelImportParser.parseBpelFromUrl(bpelURL, wsdls);
      RemoteDeployerMBean remoteDeployer = Misc.getMBeanProxy(RemoteDeployerMBean.class, jmxServiceUrl, jmxObjectName);
      processName = remoteDeployer.deploy(bpelDeployment);
    } catch (Exception e) {
      throw new OrchestraRuntimeException("Exception while deploying bpel from url : " + bpelURL, e);
    }
    return processName;
  }

  /**
   * Deploy a process
   *
   * @param line
   *            parsed command line
   */
  protected static void deploy(final CommandLine line) {
    URL bpelFile = null;
    List<URL> wsdlUrls = new ArrayList<URL>();
    String jmxServiceUrl = line.getOptionValue(ClientOptions.JMX_URL_OPTION,
        JMXClient.jmxServiceUrl);
    String jmxObjectName = line.getOptionValue(
        ClientOptions.JMX_OBJECT_OPTION, JMXClient.jmxObjectName);

    if (line.hasOption(ClientOptions.BPEL_OPTION)) {
      String bpelFileName = line.getOptionValue(ClientOptions.BPEL_OPTION);
      try {
        bpelFile = new File(bpelFileName).toURL();
      } catch (MalformedURLException e) {
        throw new OrchestraRuntimeException("Invalid bpel file name " + bpelFileName, e);
      }
    }
    if (line.hasOption(ClientOptions.WSDL_OPTION)) {
      String extWsdlCmd = line.getOptionValue(ClientOptions.WSDL_OPTION);
      for (String wsdl : extWsdlCmd.split(",")) {
        try {
          wsdlUrls.add(new File(wsdl).toURL());
        } catch (MalformedURLException e) {
          throw new OrchestraRuntimeException("Invalid wsdl file name " + wsdl, e);
        }
      }
    }

    ProcessDefinition processName = deploy(bpelFile, wsdlUrls, jmxServiceUrl,
        jmxObjectName);
    System.out.println("Process " + processName.getUUID() + " deployed !");
  }

  public boolean parseAndExecute(final String[] args) {
    // create the parser
    CommandLineParser parser = new GnuParser();
    CommandLine line = null;
    try {
      // parse the command line arguments
      line = parser.parse(options, args);
    } catch (ParseException exp) {
      // oops, something went wrong
      return false;
    }

    deploy(line);

    return true;
  }

  public Options getOptions() {
    return options;
  }

  public String getDescription() {
    return "To deploy a BPEL process";
  }

}
