/**
 * Copyright (C) 2007  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.jmxclient;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.naming.ServiceUnavailableException;

import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.jmx.JMXConfiguration;

/**
 * @author Guillaume Porcher
 */
public final class JMXClient {

  private JMXClient() {
  }

  // Default service url of the JMX remote deployer
  protected static String jmxServiceUrl;

  // Default object name of the JMX remote deployer
  protected static String jmxObjectName;

  private static void printHelp(String description, Options opts) {
    HelpFormatter formatter = new HelpFormatter();
    formatter.setSyntaxPrefix("");
    System.out.println();
    System.out.println(description + ":");
    formatter.printHelp("java -jar orchestra-jmxclient.jar", opts, true);
    System.out.println();
  }

  /**
   * Launches the JMXClient
   *
   * @param args
   */
  public static void main(String[] args) {
    try {
      try {
        jmxServiceUrl = JMXConfiguration.getInstance().getServiceUrl();
        jmxObjectName = JMXConfiguration.getInstance().getObjectName();
      } catch (Exception e) {
        // Default configuration not loaded
        throw new OrchestraRuntimeException("JMX Configuration error", e);
      }
      List<ClientAction> actions = new ArrayList<ClientAction>();
      actions.add(new DeployAction());
      actions.add(new DeployBarAction());
      actions.add(new UndeployAction());
      actions.add(new StatusAction());
      actions.add(new ListProcessAction());
      actions.add(new ListInstanceAction());
      actions.add(new GetProcessAction());
      actions.add(new LaunchAction());
      actions.add(new InitAction());

      if (System.getProperty("extraActionsClasses") != null) {
        String extraActions = System.getProperty("extraActionsClasses");
        for (String className : extraActions.split(",")) {
          try {
            Class< ? > clazz = JMXClient.class.getClassLoader().loadClass(className);
            Object o = clazz.newInstance();
            actions.add((ClientAction) o);
          } catch (Exception e) {
            throw new OrchestraRuntimeException("Error while creating action : " + className, e);
          }
        }
      }

      boolean done = false;
      for (ClientAction action : actions) {
        if (action.parseAndExecute(args)) {
          done = true;
        }
      }
      if (!done) {
        // Print Help
        CommandLineParser parser = new GnuParser();
        try {
          // parse the command line arguments
          parser.parse(ClientOptions.createHelpOptions(), args);
        } catch (ParseException exp) {
          // oops, something went wrong
          StringBuffer strBuf = new StringBuffer();
          strBuf.append("Error: invalid syntax '");
          for (String arg : args) {
            strBuf.append(" " + arg);
          }
          strBuf.append("'");
          throw new OrchestraRuntimeException(strBuf.toString());
        }

        System.out.println("JmxClient for Nova Orchestra");
        System.out.println("List of possible actions:");
        for (ClientAction action : actions) {
          printHelp(action.getDescription(), action.getOptions());
        }
      }
    } catch (OrchestraRuntimeException e) {
      Throwable cause = e.getCause();
      if ((cause instanceof IOException)
          && (cause.getCause() instanceof ServiceUnavailableException)) {
        System.out.println("Error during connection. Please check that Orchestra is started.");
      } else {
        System.out.println("Operation failed : "
            + e.getMessage());
        throw e;
      }
    }
  }
}
