package org.ow2.orchestra.jmxclient;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.orchestra.facade.def.ProcessDefinition;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.facade.jmx.RemoteDeployerMBean;
import org.ow2.orchestra.util.Misc;

public class DeployBarAction implements ClientAction {

  private final Options options = ClientOptions.createDeployBarOptions();

  /**
   * Deploy a BPEL process from a Bar
   *
   * @param barFile
   *            URL of the bar file to deploy
   * @param jmxServiceUrl
   *            url of the jmx service
   * @param jmxObjectName
   *            name of the remote deployer
   * @return QName of the deployed process
   */
  public static ProcessDefinition deployBar(final URL barFile, final String jmxServiceUrl,
      final String jmxObjectName) {
    ProcessDefinition processName = null;
    byte[] barBytes = null;
    try {
      final InputStream stream = barFile.openStream();
      final ByteArrayOutputStream output = new ByteArrayOutputStream();
      int c;
      while ((c = stream.read()) != -1) {
        output.write(c);
      }
      stream.close();
      barBytes = output.toByteArray();
    } catch (final Exception e) {
      throw new OrchestraRuntimeException(
          "couldn't read bar file from url " + barFile, e);
    }
    try {
      final RemoteDeployerMBean remoteDeployer = Misc.getMBeanProxy(RemoteDeployerMBean.class, jmxServiceUrl, jmxObjectName);
      processName = remoteDeployer.deployBar(barBytes);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException("Exception while deploying barFile : "
          + barFile, e);
    }
    return processName;
  }

  /**
   * Deploy a BPEL process from a Bar
   *
   * @param barBytes
   *            bytes array of the bar file to deploy
   * @param jmxServiceUrl
   *            url of the jmx service
   * @param jmxObjectName
   *            name of the remote deployer
   * @return QName of the deployed process
   */
  public static ProcessDefinition deployBar(final byte[] barBytes, final String jmxServiceUrl,
      final String jmxObjectName) {
    ProcessDefinition processName = null;
    try {
      final RemoteDeployerMBean remoteDeployer = Misc.getMBeanProxy(RemoteDeployerMBean.class, jmxServiceUrl, jmxObjectName);
      processName = remoteDeployer.deployBar(barBytes);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException("Exception while deploying barFile ", e);
    }
    return processName;
  }

  /**
   * Deploy a process
   *
   * @param line
   *            parsed command line
   */
  protected static void deployBar(final CommandLine line) {
    URL barFile = null;
    final String jmxServiceUrl = line.getOptionValue(ClientOptions.JMX_URL_OPTION,
        JMXClient.jmxServiceUrl);
    final String jmxObjectName = line.getOptionValue(
        ClientOptions.JMX_OBJECT_OPTION, JMXClient.jmxObjectName);

    final String barFileName = line.getOptionValue(ClientOptions.BAR_OPTION);
    try {
      barFile = new File(barFileName).toURL();
    } catch (final MalformedURLException e) {
      throw new OrchestraRuntimeException("Invalid bar url " + barFileName, e);
    }

    final ProcessDefinition processName = DeployBarAction.deployBar(barFile, jmxServiceUrl, jmxObjectName);
    System.out.println("Process " + processName.getUUID() + " deployed !");

  }

  public boolean parseAndExecute(final String[] args) {
    // create the parser
    final CommandLineParser parser = new GnuParser();
    CommandLine line = null;
    try {
      // parse the command line arguments
      line = parser.parse(this.options, args);
    } catch (final ParseException exp) {
      // oops, something went wrong
      return false;
    }

    DeployBarAction.deployBar(line);

    return true;
  }

  public Options getOptions() {
    return this.options;
  }

  public String getDescription() {
    return "To deploy a BPEL process from a Bar";
  }

}
