package org.ow2.orchestra.jmxclient;

import javax.xml.namespace.QName;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.facade.jmx.RemoteDeployerMBean;
import org.ow2.orchestra.util.Misc;

public class UndeployAction implements ClientAction {

  // create Options object
  private final Options options = ClientOptions.createUndeployOptions();

  /**
   * Undeploys a BPEL process
   *
   * @param processQName
   *            name of the process to undeploy
   * @param jmxServiceUrl
   *            url of the jmx service
   * @param jmxObjectName
   *            name of the remote deployer
   * @return <code>true</code> if the process was successfully removed.
   */
  public static boolean undeploy(final QName processQName, final String jmxServiceUrl,
      final String jmxObjectName) {
    boolean result = false;
    try {
      final RemoteDeployerMBean remoteDeployer = Misc.getMBeanProxy(RemoteDeployerMBean.class, jmxServiceUrl, jmxObjectName);
      result = remoteDeployer.undeploy(processQName);
    } catch (final Exception e) {
      throw new OrchestraRuntimeException(
          "Exception caught while undeploying process : " + processQName, e);
    }
    return result;
  }

  /**
   * Undeploys a process
   *
   * @param line
   *            parsed command line
   */
  protected static void undeploy(final CommandLine line) {
    QName processQName = null;

    final String jmxServiceUrl = line.getOptionValue(ClientOptions.JMX_URL_OPTION,
        JMXClient.jmxServiceUrl);
    final String jmxObjectName = line.getOptionValue(ClientOptions.JMX_OBJECT_OPTION,
        JMXClient.jmxObjectName);

    if (line.hasOption(ClientOptions.PROCESS_QNAME_OPTION)) {
      processQName = QName.valueOf(line
          .getOptionValue(ClientOptions.PROCESS_QNAME_OPTION));
    }
    final boolean result = UndeployAction.undeploy(processQName, jmxServiceUrl, jmxObjectName);

    if (result) {
      System.out.println("Process " + processQName + " undeployed !");
    } else {
      System.out.println("Problem during undeployment of process "
          + processQName);
    }
  }

  public boolean parseAndExecute(final String[] args) {
    // create the parser
    final CommandLineParser parser = new GnuParser();
    CommandLine line = null;
    try {
      // parse the command line arguments
      line = parser.parse(this.options, args);
    } catch (final ParseException exp) {
      // oops, something went wrong
      return false;
    }

    UndeployAction.undeploy(line);

    return true;
  }

  public Options getOptions() {
    return this.options;
  }

  public String getDescription() {
    return "To undeploy a BPEL process";
  }
}
