/**
 * Copyright (C) 2007  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.jmxclient;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.UndeclaredThrowableException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.naming.ServiceUnavailableException;

import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.ow2.orchestra.facade.exception.OrchestraRuntimeException;
import org.ow2.orchestra.util.Misc;
import org.ow2.orchestra.util.OrchestraConstants;

/**
 * @author Guillaume Porcher
 */
public final class JMXClient {

  private JMXClient() {
  }

  // Default service url of the JMX remote deployer
  static String jmxServiceUrl;

  // Default object name of the JMX remote deployer
  static String jmxObjectName;

  private static void printHelp(final String description, final Options opts) {
    final HelpFormatter formatter = new HelpFormatter();
    formatter.setSyntaxPrefix("");
    System.out.println();
    System.out.println(description + ":");
    formatter.printHelp("java -jar orchestra-jmxclient.jar", opts, true);
    System.out.println();
  }

  /**
   * Launches the JMXClient
   *
   * @param args
   */
  public static void main(final String[] args) {
    try {
      final String propertiesLocation = System.getProperty("orchestra.properties");
      InputStream orchestraPropsStream;

      if (propertiesLocation != null) {
        try {
          orchestraPropsStream = new FileInputStream(propertiesLocation);
        } catch (final FileNotFoundException e) {
          throw new OrchestraRuntimeException("Cannot load orchestra.properties from " + propertiesLocation);
        }
      } else {
        orchestraPropsStream = JMXClient.class.getClassLoader().getResourceAsStream("orchestra.properties");
        if (orchestraPropsStream == null) {
          System.out.println("orchestra.properties not found in classpath.");
          System.out.println("Please specifiy orchestra.properties location with -Dorchestra.properties=<url>.");
          return;
        }
      }
      try {
        final Properties orchestraProperties = new Properties();
        orchestraProperties.load(orchestraPropsStream);
        JMXClient.jmxServiceUrl = orchestraProperties.getProperty(OrchestraConstants.JMX_SERVICE_URL);
        JMXClient.jmxObjectName = orchestraProperties.getProperty(OrchestraConstants.JMX_OBJECT_NAME);
      } catch (final Exception e) {
        // Default configuration not loaded
        throw new OrchestraRuntimeException("Exception occured while loading orchestra.properties", e);
      } finally {
        Misc.close(orchestraPropsStream);
      }

      final List<ClientAction> actions = new ArrayList<ClientAction>();
      actions.add(new DeployAction());
      actions.add(new DeployBarAction());
      actions.add(new UndeployAction());
      actions.add(new StatusAction());
      actions.add(new ListProcessAction());
      actions.add(new ListInstanceAction());
      actions.add(new GetProcessAction());
      actions.add(new LaunchAction());
      actions.add(new InitAction());

      if (System.getProperty("extraActionsClasses") != null) {
        final String extraActions = System.getProperty("extraActionsClasses");
        for (final String className : extraActions.split(",")) {
          try {
            final Class< ? > clazz = JMXClient.class.getClassLoader().loadClass(className);
            final Object o = clazz.newInstance();
            actions.add((ClientAction) o);
          } catch (final Exception e) {
            throw new OrchestraRuntimeException("Error while creating action : " + className, e);
          }
        }
      }

      boolean done = false;
      for (final ClientAction action : actions) {
        if (action.parseAndExecute(args)) {
          done = true;
        }
      }
      if (!done) {
        // Print Help
        final CommandLineParser parser = new GnuParser();
        try {
          // parse the command line arguments
          parser.parse(ClientOptions.createHelpOptions(), args);
        } catch (final ParseException exp) {
          // oops, something went wrong
          final StringBuffer strBuf = new StringBuffer();
          strBuf.append("Error: invalid syntax '");
          for (final String arg : args) {
            strBuf.append(" " + arg);
          }
          strBuf.append("'");
          throw new OrchestraRuntimeException(strBuf.toString());
        }

        System.out.println("JmxClient for Orchestra");
        System.out.println("List of possible actions:");
        for (final ClientAction action : actions) {
          JMXClient.printHelp(action.getDescription(), action.getOptions());
        }
      }
    } catch (final OrchestraRuntimeException e) {
      Throwable cause = e.getCause();
      if (cause instanceof UndeclaredThrowableException) {
        cause = cause.getCause();
      }
      if ((cause instanceof IOException)
          && (cause.getCause() instanceof ServiceUnavailableException)) {
        System.out.println("Error during connection to " + JMXClient.jmxServiceUrl + ". Please check that Orchestra is started.");
      } else {
        System.out.println("Operation failed : "
            + e.getMessage());
        throw e;
      }
    }
  }
}
