/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.env;

import org.ow2.orchestra.pvm.env.Context;
import org.ow2.orchestra.pvm.env.EnvironmentFactory;
import org.ow2.orchestra.pvm.env.PvmEnvironmentFactory;
import org.ow2.orchestra.pvm.internal.stream.StreamSource;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireDefinition;
import org.ow2.orchestra.pvm.internal.wire.xml.WireParser;
import org.ow2.orchestra.pvm.internal.xml.Parse;
import org.ow2.orchestra.pvm.internal.xml.Parser;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * main parser. It parses the <code>contexts</code>, which is assumed the
 * document element and can contain the {@link EnvironmentFactoryXmlParser
 * environment-factory} context and the {@link EnvironmentXmlParser environment}
 * context.
 *
 * See {@link Parser} for usage instructions.
 *
 * @author Tom Baeyens
 */
public class PvmEnvironmentFactoryParser extends Parser {

  private static final long serialVersionUID = 1L;

  private WireParser environmentFactoryXmlParser = new WireParser();
  private WireParser environmentXmlParser = new WireParser();

  private static final PvmEnvironmentFactoryParser INSTANCE = new PvmEnvironmentFactoryParser();

  public static PvmEnvironmentFactoryParser getInstance() {
    return PvmEnvironmentFactoryParser.INSTANCE;
  }

  /**
   * parses the given stream source with {@link #getInstance() the current
   * parser}.
   */
  static EnvironmentFactory parse(final StreamSource streamSource) {
    final EnvironmentFactory environmentFactory = (EnvironmentFactory) PvmEnvironmentFactoryParser.getInstance()
        .createParse().setStreamSource(streamSource).execute().checkProblems(
            "environment configuration " + streamSource).getDocumentObject();
    return environmentFactory;
  }

  @Override
  public Object parseDocument(final Document document, final Parse parse) {
    final Element documentElement = document.getDocumentElement();

    // if the default environment factory was already set in the parse
    PvmEnvironmentFactory pvmEnvironmentFactory = (PvmEnvironmentFactory) parse
        .getDocumentObject();
    if (pvmEnvironmentFactory == null) {
      pvmEnvironmentFactory = new PvmEnvironmentFactory();
      parse.setDocumentObject(pvmEnvironmentFactory);
    }

    final WireDefinition environmentFactoryWireDefinition = this.getApplicationWireDefinition(
        documentElement, parse);
    final WireDefinition environmentWireDefinition = this.getBlockWireDefinition(
        documentElement, parse);

    // create the application wire context from the definition
    final WireContext environmentFactoryWireContext = new WireContext(
        environmentFactoryWireDefinition,
        Context.CONTEXTNAME_ENVIRONMENT_FACTORY);
    // propagate the parser classloader to the application context
    environmentFactoryWireContext.setClassLoader(this.classLoader);

    // configure the default environment factory
    pvmEnvironmentFactory
        .setEnvironmentFactoryCtxWireContext(environmentFactoryWireContext);
    pvmEnvironmentFactory
        .setEnvironmentCtxWireDefinition(environmentWireDefinition);

    parse.setDocumentObject(pvmEnvironmentFactory);

    return pvmEnvironmentFactory;
  }

  WireDefinition getApplicationWireDefinition(final Element documentElement,
      final Parse parse) {
    final Element applicationElement = XmlUtil.element(documentElement,
        Context.CONTEXTNAME_ENVIRONMENT_FACTORY);
    if (applicationElement != null) {
      return (WireDefinition) this.environmentFactoryXmlParser.parseDocumentElement(
          applicationElement, parse);
    }
    return null;
  }

  WireDefinition getBlockWireDefinition(final Element documentElement, final Parse parse) {
    final Element blockElement = XmlUtil.element(documentElement,
        Context.CONTEXTNAME_ENVIRONMENT);
    if (blockElement != null) {
      return (WireDefinition) this.environmentXmlParser.parseDocumentElement(
          blockElement, parse);
    }
    return null;
  }

  public WireParser getEnvironmentFactoryXmlParser() {
    return this.environmentFactoryXmlParser;
  }

  public void setEnvironmentFactoryXmlParser(final WireParser applicationWireXmlParser) {
    this.environmentFactoryXmlParser = applicationWireXmlParser;
  }

  public WireParser getEnvironmentXmlParser() {
    return this.environmentXmlParser;
  }

  public void setEnvironmentXmlParser(final WireParser blockWireXmlParser) {
    this.environmentXmlParser = blockWireXmlParser;
  }
}
