/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.job;

import java.text.SimpleDateFormat;
import java.util.Date;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.env.Transaction;
import org.ow2.orchestra.pvm.internal.cal.BusinessCalendar;
import org.ow2.orchestra.pvm.internal.cal.Duration;
import org.ow2.orchestra.pvm.internal.env.JobContext;
import org.ow2.orchestra.pvm.internal.jobexecutor.JobAddedNotification;
import org.ow2.orchestra.pvm.internal.jobexecutor.JobDbSession;
import org.ow2.orchestra.pvm.internal.jobexecutor.JobExecutor;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.util.Clock;
import org.ow2.orchestra.pvm.job.Timer;
import org.ow2.orchestra.pvm.model.ObservableElement;

/**
 * a runtime timer instance.
 *
 * <p>
 * Timers can be
 * </p>
 *
 * @author Tom Baeyens
 * @author Pascal Verdage
 * @author Alejandro Guizar
 */
public class TimerImpl extends JobImpl<Boolean> implements Timer {

  private static final long serialVersionUID = 1L;
  private static final Log LOG = Log.getLog(TimerImpl.class.getName());

  private static final String DATE_FORMAT = "yyyy-MM-dd HH:mm:ss,SSS";

  protected String signalName;
  protected String eventName;
  protected String repeat;

  public static final String EVENT_TIMER = "timer";

  public TimerImpl() {
  }

  public void setDueDateDescription(final String dueDateDescription) {
    final Duration duration = new Duration(dueDateDescription);
    final Date now = this.getDueDate();

    if (duration.isBusinessTime() || duration.getMonths() > 0
        || duration.getYears() > 0) {
      final Environment environment = Environment.getCurrent();
      if (environment == null) {
        throw new PvmException(
            "no environment to get business calendar for calculating dueDate "
                + dueDateDescription);
      }
      final BusinessCalendar businessCalendar = environment
          .get(BusinessCalendar.class);
      this.dueDate = businessCalendar.add(now, duration);

    } else {
      final long millis = duration.getMillis()
          + 1000
          * (duration.getSeconds() + 60 * (duration.getMinutes() + 60 * (duration
              .getHours() + 24 * (duration.getDays() + 7 * duration.getWeeks()))));
      this.dueDate = new Date(now.getTime() + millis);
    }
  }

  public Boolean execute(final Environment environment) throws Exception {
    if (TimerImpl.LOG.isDebugEnabled()) {
      TimerImpl.LOG.debug("executing " + this);
    }

    if (environment == null) {
      throw new PvmException("environment is null");
    }

    final JobContext jobContext = new JobContext(this);
    environment.addContext(jobContext);
    try {
      if (this.signalName != null) {
        if (TimerImpl.LOG.isDebugEnabled()) {
          TimerImpl.LOG
              .debug("feeding timer signal " + this.signalName + " into "
                  + this.execution);
        }
        this.execution.signal(this.signalName);
      }

      if (this.eventName != null) {
        final ObservableElement eventSource = this.execution.getNode();
        if (TimerImpl.LOG.isDebugEnabled()) {
          TimerImpl.LOG.debug("firing event " + this.signalName + " into " + eventSource);
        }
        this.execution.fire(this.eventName, eventSource);
      }

    } finally {
      environment.removeContext(jobContext);
    }

    boolean deleteThisJob = true;
    // if there is no repeat on this timer
    if (this.repeat == null) {
      // delete the jobImpl
      if (TimerImpl.LOG.isDebugEnabled()) {
        TimerImpl.LOG.debug("deleting " + this);
      }
      final JobDbSession dbSession = environment.get(JobDbSession.class);
      if (dbSession == null) {
        throw new PvmException("no " + JobDbSession.class.getName()
            + " in environment");
      }
      dbSession.delete(this);

    } else { // there is a repeat on this timer
      deleteThisJob = false;
      // suppose that it took the timer runner thread a very long time to
      // execute the timers
      // then the repeat action dueDate could already have passed
      do {
        this.setDueDateDescription(this.repeat);
      } while (this.dueDate.getTime() <= Clock.getCurrentTime().getTime());

      if (TimerImpl.LOG.isDebugEnabled()) {
        TimerImpl.LOG.debug("rescheduled " + this + " for " + TimerImpl.formatDueDate(this.dueDate));
      }

      // release the lock on the timer
      this.setLockOwner(null);
      this.setLockExpirationTime(null);

      // notify the jobExecutor at the end of the transaction
      final JobExecutor jobExecutor = environment.get(JobExecutor.class);
      if (jobExecutor != null) {
        final Transaction transaction = environment.get(Transaction.class);
        if (transaction == null) {
          throw new PvmException("no transaction in environment");
        }
        final JobAddedNotification jobNotificator = new JobAddedNotification(
            jobExecutor);
        transaction.registerSynchronization(jobNotificator);
      }
    }

    return deleteThisJob;
  }

  @Override
  public String toString() {
    final StringBuffer buffer = new StringBuffer();
    buffer.append("timer[");
    buffer.append(this.dbid);
    if (this.dueDate != null) {
      buffer.append("|");
      buffer.append(TimerImpl.formatDueDate(this.dueDate));
    }
    if (this.signalName != null) {
      buffer.append("|");
      buffer.append(this.signalName);
    }
    if (this.eventName != null) {
      buffer.append("|");
      buffer.append(this.eventName);
    }
    buffer.append("]");
    return buffer.toString();
  }

  public static String formatDueDate(final Date date) {
    return new SimpleDateFormat(TimerImpl.DATE_FORMAT).format(date);
  }

  public String getSignalName() {
    return this.signalName;
  }

  public void setSignalName(final String signalName) {
    this.signalName = signalName;
  }

  public String getEventName() {
    return this.eventName;
  }

  public void setEventName(final String eventName) {
    this.eventName = eventName;
  }

  public String getRepeat() {
    return this.repeat;
  }

  public void setRepeat(final String repeat) {
    this.repeat = repeat;
  }
}
