/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.jobexecutor;

import java.util.Date;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.env.Transaction;
import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.pvm.internal.job.JobImpl;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.job.Job;

/**
 * @author Tom Baeyens
 */
public class ExecuteJobCmd implements Command<Job> {

  private static final long serialVersionUID = 1L;

  private static final Log LOG = Log.getLog(ExecuteJobCmd.class.getName());

  protected long jobDbid;

  public ExecuteJobCmd(final long jobDbid) {
    this.jobDbid = jobDbid;
  }

  public Job execute(final Environment environment) throws Exception {
    final JobDbSession session = environment.get(JobDbSession.class);
    if (session == null) {
      throw new PvmException("no job session configured to get job " + this.jobDbid
          + " for execution");
    }
    final JobImpl< ? > job = session.get(JobImpl.class, this.jobDbid);

    // in case of exclusive jobs, the jobImpl might have been deleted
    // before we execute it (they are in a list)
    if (job != null && !job.isSuspended()) {
      ExecuteJobCmd.LOG.debug("executing job " + job + "...");
      job.execute(environment);
      ExecuteJobCmd.LOG.debug("executed job " + job);

      // if this job is locked too long, it could be unlocked by the
      // lockmonitor and
      // executed by another thread.
      final Date lockExpirationDate = job.getLockExpirationTime();
      // can be null if it was rescheduled
      if (lockExpirationDate != null) {
        final long lockExpiration = lockExpirationDate.getTime();
        final long currentTime = System.currentTimeMillis();
        if (currentTime > lockExpiration) {
          throw new PvmException("job took too long: lock expired "
              + (currentTime - lockExpiration) + "ms ago");
        }
      }
    } else if (job != null && job.isSuspended()) {
      ExecuteJobCmd.LOG.debug("job " + this.jobDbid + " suspended");
      // release the lock on the job
      job.setLockOwner(null);
      job.setLockExpirationTime(null);
    }  else {
      ExecuteJobCmd.LOG.debug("job " + this.jobDbid + " no longer exists");
    }
    return job;
  }

  /**
   * This transaction will be marked for rollback. A command will be associated
   * with the Transaction.EVENT_AFTERCOMPLETION (after the jobImpl locks of the
   * current transaction are released). Then the command will update the JobImpl
   * with the exception details in a separate transaction.
   */
  protected void handleJobExecutionException(final Environment environment,
      final JobImpl< ? > job, final Throwable exception) {
    final Transaction transaction = environment.get(Transaction.class);
    transaction.setRollbackOnly();

    final CommandService commandService = environment
        .get(CommandService.class);
    final JobExceptionHandler jobExceptionHandler = new JobExceptionHandler(job
        .getDbid(), exception, commandService);
    transaction.registerSynchronization(jobExceptionHandler);
  }
}
