/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.tx;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.transaction.Synchronization;

import org.ow2.orchestra.pvm.env.Transaction;
import org.ow2.orchestra.pvm.internal.log.Log;

/**
 * simple 2 phase commit transaction. no logging or recovery. non thread safe
 * (which is ok).
 *
 * @author Tom Baeyens
 */
public class StandardTransaction implements Transaction, Serializable {

  private static final long serialVersionUID = 1L;
  private static Log log = Log.getLog(StandardTransaction.class.getName());

  enum State {
    CREATED, ACTIVE, ROLLBACKONLY, COMMITTED, ROLLEDBACK
  }

  protected List<StandardResource> resources;
  protected List<StandardSynchronization> synchronizations;
  protected State state = State.CREATED;

  // methods for interceptor //////////////////////////////////////////////////

  public void begin() {
    StandardTransaction.log.debug("beginning " + this);
    this.state = State.ACTIVE;
  }

  public void complete() {
    if (this.state == State.ACTIVE) {
      this.commit();
    } else if (this.state == State.ROLLBACKONLY) {
      this.rollback();
    } else {
      throw new TransactionException("complete on transaction in state "
          + this.state);
    }
  }

  // public tx methods ////////////////////////////////////////////////////////

  public void setRollbackOnly() {
    if (this.state != State.ACTIVE) {
      throw new TransactionException("transaction was not active: " + this.state);
    }
    this.state = State.ROLLBACKONLY;
  }

  public boolean isRollbackOnly() {
    return (this.state == State.ROLLBACKONLY) || (this.state == State.ROLLEDBACK);
  }

  // commit ///////////////////////////////////////////////////////////////////

  /** implements simplest two phase commit. */
  public void commit() {
    if (this.state != State.ACTIVE) {
      throw new TransactionException("commit on transaction in state " + this.state);
    }

    StandardTransaction.log.trace("committing " + this);

    try {
      this.beforeCompletion();

      if (this.resources != null) {
        // prepare
        // //////////////////////////////////////////////////////////////
        // the prepare loop will be skipped at the first exception
        for (final StandardResource standardResource : this.resources) {
          StandardTransaction.log.trace("preparing resource " + standardResource);
          standardResource.prepare();
        }
      }

      // for any exception in the prepare phase, we'll rollback
    } catch (final RuntimeException exception) {
      try {
        StandardTransaction.log.debug("resource threw exception in prepare.  rolling back.");
        this.rollbackResources();
      } catch (final Exception rollbackException) {
        StandardTransaction.log.error("rollback failed as well", rollbackException);
      }

      // rethrow
      throw exception;
    }

    // here is the point of no return :-)

    // commit ///////////////////////////////////////////////////////////////
    Throwable commitException = null;
    if (this.resources != null) {
      // The commit loop will try to send the commit to every resource,
      // No matter what it takes. If exceptions come out of resource.commit's
      // they will be suppressed and the first exception will be rethrown after
      // all the resources are commited
      for (final StandardResource standardResource : this.resources) {
        try {
          StandardTransaction.log.trace("committing resource " + standardResource);
          standardResource.commit();

          // Exceptions in the commit phase will not lead to rollback, since
          // some resources
          // might have committed and can't go back.
        } catch (final Throwable t) {
          // TODO this should go to a special log for sys admin recovery
          StandardTransaction.log.error("commit failed for resource " + standardResource, t);
          if (commitException == null) {
            commitException = t;
          }
        }
      }
    }

    this.state = State.COMMITTED;
    this.afterCompletion();
    StandardTransaction.log.debug("committed " + this);

    if (commitException != null) {
      if (commitException instanceof RuntimeException) {
        throw (RuntimeException) commitException;
      } else if (commitException instanceof Error) {
        throw (Error) commitException;
      }
      throw new TransactionException("resource failed to commit",
          commitException);
    }
  }

  // rollback /////////////////////////////////////////////////////////////////

  public void rollback() {
    if ((this.state != State.ACTIVE) && (this.state != State.ROLLBACKONLY)) {
      throw new TransactionException("rollback on transaction in state "
          + this.state);
    }

    StandardTransaction.log.trace("rolling back " + this);

    this.beforeCompletion();
    this.rollbackResources();
  }

  void rollbackResources() {
    if (this.resources != null) {
      for (final StandardResource resource : this.resources) {
        try {
          StandardTransaction.log.trace("rolling back resource " + resource);
          resource.rollback();
        } catch (final RuntimeException e) {
          StandardTransaction.log.error("rollback failed for resource " + resource);
        }
      }
    }

    this.state = State.ROLLEDBACK;

    this.afterCompletion();

    StandardTransaction.log.debug("rolled back");
  }

  // synchronizations /////////////////////////////////////////////////////////

  public void registerSynchronization(final Synchronization synchronization) {
    if (this.synchronizations == null) {
      this.synchronizations = new ArrayList<StandardSynchronization>();
    }
    this.synchronizations.add(new StandardSynchronization(synchronization));
  }

  public void afterCompletion() {
    if (this.synchronizations != null) {
      for (final StandardSynchronization synchronization : this.synchronizations) {
        synchronization.afterCompletion(this.state);
      }
    }
  }

  public void beforeCompletion() {
    if (this.synchronizations != null) {
      for (final StandardSynchronization synchronization : this.synchronizations) {
        synchronization.beforeCompletion();
      }
    }
  }

  // resource enlisting ///////////////////////////////////////////////////////

  public void enlistResource(final StandardResource standardResource) {
    if (this.resources == null) {
      this.resources = new ArrayList<StandardResource>();
    }
    StandardTransaction.log.trace("enlisting resource " + standardResource
        + " to standard transaction");
    this.resources.add(standardResource);
  }

  List<StandardResource> getResources() {
    return this.resources;
  }

  @Override
  public String toString() {
    return "StandardTransaction[" + System.identityHashCode(this) + "]";
  }
}
