/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.wire.binding;

import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.stream.UrlStreamSource;
import org.ow2.orchestra.pvm.internal.util.ReflectUtil;
import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.DeployerManagerDescriptor;
import org.ow2.orchestra.pvm.internal.wire.xml.WireParser;
import org.ow2.orchestra.pvm.internal.xml.Parse;
import org.ow2.orchestra.pvm.internal.xml.Parser;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Element;

/**
 * parses a descriptor for creating a {@link org.ow2.orchestra.pvm.internal.deploy.DeployerManager}.
 *
 * See schema docs for more details.
 *
 * @author Tom Baeyens
 */
public class DeployerManagerBinding extends WireDescriptorBinding {

  private static final Log LOG = Log.getLog(DeployerManagerBinding.class.getName());

  public DeployerManagerBinding() {
    super("deployer-manager");
  }

  public Object parse(final Element element, final Parse parse, final Parser parser) {
    final DeployerManagerDescriptor descriptor = new DeployerManagerDescriptor();

    if (element.hasAttribute("resource")) {
      final String resource = element.getAttribute("resource");
      try {
        final Enumeration<URL> enumeration = ReflectUtil.getResources(parse
            .getClassLoader(), resource);
        if (enumeration != null) {
          while (enumeration.hasMoreElements()) {
            final URL url = enumeration.nextElement();
            DeployerManagerBinding.LOG.trace("importing language deployers from " + url);
            parser.importStream(new UrlStreamSource(url), element, parse);
          }
        }
      } catch (final Exception e) {
        parse.addProblem("couldn't parse language deployer resource '"
            + resource + "'", e);
      }
    }

    final List<Element> languageElements = XmlUtil.elements(element, "language");
    if (languageElements != null) {
      for (final Element languageElement : languageElements) {
        String language = null;
        if (languageElement.hasAttribute("name")) {
          language = languageElement.getAttribute("name");
        } else {
          parse.addProblem("name is a required attribute in element language: "
              + XmlUtil.toString(languageElement));
        }

        final String extension = null;
        if (element.hasAttribute("extension")) {
          language = element.getAttribute("extension");
        }

        final List<Descriptor> deployerDescriptors = new ArrayList<Descriptor>();

        final List<Element> deployerElements = XmlUtil.elements(languageElement);
        if (deployerElements != null) {
          for (final Element deployerElement : deployerElements) {
            final Descriptor deployerDescriptor = (Descriptor) parser.parseElement(
                deployerElement, parse, WireParser.CATEGORY_DESCRIPTOR);
            deployerDescriptors.add(deployerDescriptor);
          }
        }

        if (deployerDescriptors.isEmpty()) {
          parse.addProblem("no deployers defined for " + language);
        }

        descriptor.addLanguage(language, extension, deployerDescriptors);
      }
    } else {
      parse
          .addProblem("at least one language is required in a deployer-manager: "
              + XmlUtil.toString(element));
    }

    return descriptor;
  }

}
