package org.ow2.orchestra.pvm.internal.wire.descriptor;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireException;

/**
 *
 * <p>
 * This {@link Descriptor} creates a {@link Map}.
 * </p>
 *
 * <p>
 * If no specific implementation for the {@link Map} is specified, a
 * {@link HashMap} will be used.
 * </p>
 *
 * <p>
 * Entries can be added during the map initialization. The list of entries (key,
 * value) to add must be specified with two operations:
 * <ol>
 * <li>{@link #setKeyDescriptors(List)} sets the list of the keys to generate.</li>
 * <li>{@link #setValueDescriptors(List)} sets the list of value associated with
 * these keys.</li>
 * </ol>
 * The two lists must be in the same order (the n-th element of the key list
 * will be associated with the n-th element of the value list).
 * </p>
 *
 * @author Tom Baeyens
 * @author Guillaume Porcher (documentation)
 *
 * @see Descriptor
 */
public class MapDescriptor extends CollectionDescriptor {

  private static final long serialVersionUID = 1L;

  private List<Descriptor> keyDescriptors;

  public MapDescriptor() {
    super(HashMap.class.getName());
  }

  @Override
  public void initialize(final Object object, final WireContext wireContext) {
    final Map<Object, Object> map = (Map<Object, Object>) object;
    try {
      if (this.keyDescriptors != null) {
        for (int i = 0; i < this.keyDescriptors.size(); i++) {
          final Descriptor keyDescriptor = this.keyDescriptors.get(i);
          final Descriptor valueDescriptor = this.valueDescriptors.get(i);
          final Object key = wireContext.create(keyDescriptor, true);
          final Object value = wireContext.create(valueDescriptor, true);
          map.put(key, value);
        }
      }
    } catch (final WireException e) {
      throw e;
    } catch (final Exception e) {
      throw new WireException("couldn't initialize object '"
          + (this.name != null ? this.name : this.className) + "'", e);
    }
  }

  public List<Descriptor> getKeyDescriptors() {
    return this.keyDescriptors;
  }

  public void setKeyDescriptors(final List<Descriptor> keyDescriptors) {
    this.keyDescriptors = keyDescriptors;
  }
}
