/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.wire.descriptor;

import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireDefinition;
import org.ow2.orchestra.pvm.internal.wire.WireException;

/**
 * @author Tom Baeyens
 */
public class PropertiesDescriptor extends MapDescriptor {

  private static final long serialVersionUID = 1L;

  protected String url;
  protected String file;
  protected String resource;
  protected boolean isXml;

  public PropertiesDescriptor() {
    this.className = Properties.class.getName();
  }

  @Override
  public Object construct(final WireContext wireContext) {
    return new Properties();
  }

  @Override
  public void initialize(final Object object, final WireContext wireContext) {
    String description = null;
    try {
      if (this.url != null) {
        description = "url " + this.url;
        final InputStream inputStream = new URL(this.url).openStream();
        this.load(object, inputStream);
      }

      if (this.file != null) {
        description = "file " + this.file;
        final InputStream inputStream = new FileInputStream(this.file);
        this.load(object, inputStream);
      }

      if (this.resource != null) {
        description = "resource " + this.resource;
        final InputStream inputStream = wireContext.getClassLoader()
            .getResourceAsStream(this.resource);
        if (inputStream == null) {
          throw new RuntimeException("resource " + this.resource + " doesn't exist");
        }
        this.load(object, inputStream);
      }

    } catch (final Exception e) {
      throw new WireException("couldn't read properties from " + description, e);
    }

    super.initialize(object, wireContext);
  }

  @Override
  public Class< ? > getType(final WireDefinition wireDefinition) {
    return Properties.class;
  }

  protected void load(final Object object, final InputStream inputStream) throws Exception {
    final Properties properties = (Properties) object;
    if (this.isXml) {
      properties.loadFromXML(inputStream);
    } else {
      properties.load(inputStream);
    }
  }

  public String getFile() {
    return this.file;
  }

  public void setFile(final String file) {
    this.file = file;
  }

  public String getResource() {
    return this.resource;
  }

  public void setResource(final String resource) {
    this.resource = resource;
  }

  public String getUrl() {
    return this.url;
  }

  public void setUrl(final String url) {
    this.url = url;
  }

  public boolean isXml() {
    return this.isXml;
  }

  public void setXml(final boolean isXml) {
    this.isXml = isXml;
  }
}
