package org.ow2.orchestra.pvm.internal.wire.operation;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.ow2.orchestra.pvm.internal.util.ReflectUtil;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireException;
import org.ow2.orchestra.pvm.internal.wire.descriptor.ArgDescriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.ObjectDescriptor;

/**
 * invokes a method.
 *
 * @author Tom Baeyens
 * @author Guillaume Porcher (documentation)
 * @see ArgDescriptor
 * @see ObjectDescriptor
 */

public class InvokeOperation extends AbstractOperation {

  private static final long serialVersionUID = 1L;

  /** name of the method to invoke. */
  private String methodName = null;
  /** list of descriptors for creating arguments supplied to the method */
  private List<ArgDescriptor> argDescriptors = null;

  public void apply(final Object target, final WireContext wireContext) {
    try {
      final Object[] args = ObjectDescriptor.getArgs(wireContext, this.argDescriptors);
      final Class< ? > clazz = target.getClass();
      final Method method = ReflectUtil.findMethod(clazz, this.methodName, this.argDescriptors,
          args);
      if (method == null) {
        throw new WireException("method "
            + ReflectUtil.getSignature(this.methodName, this.argDescriptors, args)
            + " unavailable");
      }
      ReflectUtil.invoke(method, target, args);
    } catch (final WireException e) {
      throw e;
    } catch (final Exception e) {
      throw new WireException("couldn't invoke method " + this.methodName + ": "
          + e.getMessage(), e);
    }
  }

  /**
   * Adds a descriptor to the list of arguments descriptors.
   */
  public void addArgDescriptor(final ArgDescriptor argDescriptor) {
    if (this.argDescriptors == null) {
      this.argDescriptors = new ArrayList<ArgDescriptor>();
    }
    this.argDescriptors.add(argDescriptor);
  }

  /**
   * Gets the name of the method to invoke.
   */
  public String getMethodName() {
    return this.methodName;
  }

  /**
   * Sets the name of the method to invoke.
   *
   * @param methodName
   *          the name of the method to invoke.
   */
  public void setMethodName(final String methodName) {
    this.methodName = methodName;
  }

  /**
   * Gets the list of descriptor to create arguments supplied to the method .
   */
  public List<ArgDescriptor> getArgDescriptors() {
    return this.argDescriptors;
  }

  /**
   * Sets the list of descriptor to create arguments supplied to the method .
   */
  public void setArgDescriptors(final List<ArgDescriptor> argDescriptors) {
    this.argDescriptors = argDescriptors;
  }
}
