/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.hibernate;

import org.hibernate.cfg.ImprovedNamingStrategy;
import org.hibernate.util.StringHelper;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.util.ReflectUtil;

/**
 * @author Tom Baeyens
 */
public class PvmNamingStrategy extends ImprovedNamingStrategy {

  private static final long serialVersionUID = 1L;
  private static final Log LOG = Log.getLog(PvmNamingStrategy.class.getName());

  private String prefix;

  public PvmNamingStrategy(final String prefix) {
    if (!"".equals(prefix)) {
      this.prefix = prefix;
    }
  }

  // TABLE NAMES //////////////////////////////////////////////////////////////

  @Override
  public String classToTableName(final String className) {
    String name = ReflectUtil.getUnqualifiedClassName(className);
    if (name.endsWith("Impl")) {
      name = name.substring(0, name.length() - 4);
    }
    name = this.prefix + name.toUpperCase();
    if (PvmNamingStrategy.LOG.isTraceEnabled()) {
      PvmNamingStrategy.LOG.trace("classToTableName(" + className + ") ==> " + name);
    }
    return name;
  }

  @Override
  public String tableName(final String tableName) {
    if (this.prefix != null) {
      return this.prefix + tableName;
    }
    if (PvmNamingStrategy.LOG.isTraceEnabled()) {
      PvmNamingStrategy.LOG.trace("tableName(" + tableName + ") ==> " + tableName);
    }
    return tableName;
  }

  @Override
  public String collectionTableName(final String ownerEntity,
      final String ownerEntityTable, final String associatedEntity,
      final String associatedEntityTable, final String propertyName) {
    String name = StringHelper.unqualify(propertyName);
    name = this.prefix + name.toUpperCase() + "_";
    if (PvmNamingStrategy.LOG.isTraceEnabled()) {
      PvmNamingStrategy.LOG.trace("collectionTableName(" + ownerEntity + ", " + ownerEntityTable
          + ", " + associatedEntity + ", " + associatedEntityTable + ", "
          + propertyName + ") ==> " + name);
    }
    return name;
  }

  // COLUMN NAMES /////////////////////////////////////////////////////////////

  @Override
  public String propertyToColumnName(final String propertyName) {
    String name = propertyName;

    name = StringHelper.unqualify(name);
    name = name.toUpperCase() + "_";

    if (PvmNamingStrategy.LOG.isTraceEnabled()) {
      PvmNamingStrategy.LOG.trace("propertyToColumnName(" + propertyName + ") ==> " + name);
    }

    return name;
  }

  @Override
  public String logicalColumnName(final String columnName, final String propertyName) {
    String name = columnName != null ? columnName : propertyName;
    name = name.replace('.', '_');
    name = name.toUpperCase();
    if (!name.endsWith("_")) {
      name = name + "_";
    }
    if (PvmNamingStrategy.LOG.isTraceEnabled()) {
      PvmNamingStrategy.LOG.trace("logicalColumnName(" + columnName + ", " + propertyName
          + ") ==> " + name);
    }
    return name;
  }

  @Override
  public String columnName(final String columnName) {
    final String name = columnName.toUpperCase();
    if (PvmNamingStrategy.LOG.isTraceEnabled()) {
      PvmNamingStrategy.LOG.trace("columnName(" + columnName + ") ==> " + name);
    }
    return name;
  }

  @Override
  public String foreignKeyColumnName(final String propertyName,
      final String propertyEntityName, final String propertyTableName,
      final String referencedColumnName) {
    final String name = super.foreignKeyColumnName(propertyName, propertyEntityName,
        propertyTableName, referencedColumnName);
    if (PvmNamingStrategy.LOG.isTraceEnabled()) {
      PvmNamingStrategy.LOG.trace("foreignKeyColumnName(" + propertyName + ", "
          + propertyEntityName + ", " + propertyTableName + ", "
          + referencedColumnName + ") ==> " + name);
    }
    return name;
  }

  @Override
  public String logicalCollectionColumnName(final String columnName,
      final String propertyName, final String referencedColumn) {
    final String name = super.logicalCollectionColumnName(columnName, propertyName,
        referencedColumn);
    if (PvmNamingStrategy.LOG.isTraceEnabled()) {
      PvmNamingStrategy.LOG.trace("logicalCollectionColumnName(" + columnName + ", "
          + propertyName + ", " + referencedColumn + ") ==> " + name);
    }
    return name;
  }

  @Override
  public String logicalCollectionTableName(final String tableName,
      final String ownerEntityTable, final String associatedEntityTable, final String propertyName) {
    final String name = super.logicalCollectionTableName(tableName, ownerEntityTable,
        associatedEntityTable, propertyName);
    if (PvmNamingStrategy.LOG.isTraceEnabled()) {
      PvmNamingStrategy.LOG.trace("logicalCollectionTableName(" + tableName + ", "
          + ownerEntityTable + ", " + associatedEntityTable + ", "
          + propertyName + ") ==> " + name);
    }
    return name;
  }

  @Override
  public String joinKeyColumnName(final String joinedColumn, final String joinedTable) {
    final String name = super.joinKeyColumnName(joinedColumn, joinedTable);
    if (PvmNamingStrategy.LOG.isTraceEnabled()) {
      PvmNamingStrategy.LOG.trace("joinKeyColumnName(" + joinedColumn + ", " + joinedTable
          + ") ==> " + name);
    }
    return name;
  }
}
