/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.jobexecutor;

import java.io.PrintWriter;
import java.io.StringWriter;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.env.Transaction;
import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.job.JobImpl;
import org.ow2.orchestra.pvm.internal.log.Log;

/**
 * @author Tom Baeyens
 */
public class JobExceptionHandler implements Command<Object> {

  private static final Log LOG = Log.getLog(JobExceptionHandler.class.getName());
  private static final long serialVersionUID = 1L;

  protected long jobDbid;
  protected JobImpl< ? > job;
  protected Throwable exception;

  public JobExceptionHandler(final long jobDbid, final Throwable exception) {
    this.jobDbid = jobDbid;
    this.exception = exception;
  }

  public JobExceptionHandler(final JobImpl< ? > job, final Throwable exception) {
    this.jobDbid = job.getDbid();
    this.job = job;
    this.exception = exception;
  }

  public Object execute(final Environment environment) {
    JobExceptionHandler.LOG.debug("handling jobImpl " + this.jobDbid + " exception: "
        + this.exception.getMessage());

    if (this.job == null) {
      // load the jobImpl from the db
      final JobDbSession session = environment.get(JobDbSession.class);
      if (session == null) {
        throw new PvmException("no job-session configured to handle job");
      }
      this.job = session.get(JobImpl.class, this.jobDbid);
    }
    // serialize the stack trace
    final StringWriter sw = new StringWriter();
    this.exception.printStackTrace(new PrintWriter(sw));
    if (this.job != null) {
      // decrement the number of retries
      final int decrementedRetries = this.job.getRetries() - 1;
      JobExceptionHandler.LOG
          .debug("decrementing retries to " + decrementedRetries + " for "
              + this.job);
      this.job.setRetries(decrementedRetries);
      this.job.setException(sw.toString());
      JobExceptionHandler.LOG.debug("unlocking " + this.job);
      this.job.setLockOwner(null);
      this.job.setLockExpirationTime(null);

      final JobExecutor jobExecutor = environment.get(JobExecutor.class);
      if (decrementedRetries == 0) {
        final DeadJobHandler deadJobHandler = jobExecutor.getDeadJobHandler();
        if (deadJobHandler != null) {
          deadJobHandler.handleDeadJob(this.job);
        }
      } else {
        // notify the job executor after the transaction is completed
        final Transaction transaction = environment.get(Transaction.class);
        if ((transaction != null) && (jobExecutor != null)) {
          JobExceptionHandler.LOG.trace("registering job executor notifier with " + transaction);
          transaction.registerSynchronization(new JobAddedNotification(
              jobExecutor));
        }
      }
    }
    return null;
  }
}
