/**
 * Copyright (C) 2007  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.pvm.internal.jobexecutor.jdk;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.internal.jobexecutor.AbstractJobExecutor;
import org.ow2.orchestra.pvm.internal.jobexecutor.AcquireJobsCmd;
import org.ow2.orchestra.pvm.internal.jobexecutor.GetNextDueDateCmd;
import org.ow2.orchestra.pvm.internal.jobexecutor.JobExecutor;
import org.ow2.orchestra.pvm.internal.log.Log;

/**
 * Implementation of the {@link JobExecutor} which uses java {@link ExecutorService}
 * to execute the jobs.
 *
 * @author Guillaume Porcher
 */
public class JdkJobExecutor extends AbstractJobExecutor implements JobExecutor {
  private static final Log LOG = Log.getLog(JdkJobExecutor.class.getName());

  // runtime state
  private ExecutorService executorService;

  private JdkDispatcherThread dispatcherThread = null;

  /**
   * starts the {@link JdkDispatcherThread} and {@link ExecutorService}s
   */
  @Override
  public synchronized void start() {
    if (this.commandService == null) {
      throw new PvmException(
          "no command executor available in jobImpl executor");
    }
    if (!this.isActive) {
      this.acquireJobsCommand = new AcquireJobsCmd(this.getName(), this.getLockMillis());
      this.nextDueDateCommand = new GetNextDueDateCmd();

      this.isActive = true;
      JdkJobExecutor.LOG.trace("starting jobImpl executor threads for jobImpl executor '"
          + this.name + "'...");
      this.executorService = Executors.newCachedThreadPool();

      JdkJobExecutor.LOG.trace("starting dispatcher thread for jobImpl executor '" + this.name
          + "'...");
      this.dispatcherThread = new JdkDispatcherThread(this);
      this.dispatcherThread.start();

    } else {
      JdkJobExecutor.LOG.trace("ignoring start: jobImpl executor '" + this.name
          + "' is already started'");
    }
  }

  @Override
  public synchronized void stop() {
    this.stop(false);
  }

  @Override
  public synchronized void stop(final boolean join) {
    JdkJobExecutor.LOG.debug("stopping jobImpl executor");
    if (this.isActive) {
      this.isActive = false;
      this.dispatcherThread.deactivate(true);

      this.executorService.shutdown();
      if (join) {
        this.waitTillQueueEmpty();
      }

    } else {
      JdkJobExecutor.LOG.trace("ignoring stop: jobImpl executor '" + this.name + "' not started");
    }
  }

  protected void waitTillQueueEmpty() {
    while (!this.executorService.isTerminated()) {
      JdkJobExecutor.LOG.trace("waiting for jobImpl-id-queue to become empty");
      try {
        Thread.sleep(200);
      } catch (final InterruptedException e) {
        JdkJobExecutor.LOG
            .trace("waiting for jobImpl-id-queue to become empty got interrupted");
      }
    }
  }


  @Override
  public JdkDispatcherThread getDispatcherThread() {
    return this.dispatcherThread;
  }


  /**
   * @return the executorService
   */
  public ExecutorService getExecutorService() {
    return this.executorService;
  }
}
