/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.jobexecutor.pvm;

import java.util.Collection;
import java.util.concurrent.BlockingQueue;

import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.pvm.internal.jobexecutor.ExecuteJobCmd;
import org.ow2.orchestra.pvm.internal.jobexecutor.JobExceptionHandler;
import org.ow2.orchestra.pvm.internal.log.Log;

/**
 * @author Tom Baeyens, Guillaume Porcher
 */
public class JobExecutorThread extends Thread {

  private static final Log LOG = Log.getLog(JobExecutorThread.class.getName());

  private final PvmJobExecutor jobExecutor;
  private boolean isActive = true;

  public JobExecutorThread(final String name, final PvmJobExecutor jobExecutor) {
    super(name);
    this.jobExecutor = jobExecutor;
  }

  @Override
  public void run() {
    JobExecutorThread.LOG.info("starting...");
    try {
      final BlockingQueue<Collection<Long>> queue = this.jobExecutor.getJobDbidsQueue();
      if (queue == null) {
        JobExecutorThread.LOG.debug("no queue to take jobs from");
      } else {
        while (this.isActive) {
          try {
            JobExecutorThread.LOG.trace("taking jobs from queue");
            Collection<Long> jobDbids = null;
            jobDbids = queue.take();
            JobExecutorThread.LOG.debug("took job(s) " + jobDbids + " from queue");

            for (final Long jobDbid : jobDbids) {
              final CommandService commandService = this.jobExecutor.getCommandExecutor();
              try {
                commandService.execute(new ExecuteJobCmd(jobDbid));
              } catch (final Throwable e) {
                JobExecutorThread.LOG.error("exception occured in job " + jobDbid, e);
                commandService.execute(new JobExceptionHandler(jobDbid, e));
              }
            }
          } catch (final InterruptedException e) {
            JobExecutorThread.LOG.trace("waiting for acquired jobs got interrupted");
          } catch (final Exception e) {
            JobExecutorThread.LOG.error("exception in job executor thread", e);
          }
        }
      }
    } catch (final Throwable t) {
      t.printStackTrace();
    } finally {
      JobExecutorThread.LOG.info(this.getName() + " leaves cyberspace");
    }
  }

  public void deactivate() {
    this.stop(false);
  }

  public void stop(final boolean join) {
    if (this.isActive) {
      JobExecutorThread.LOG.trace("deactivating " + this.getName());
      this.isActive = false;
      this.interrupt();
      if (join) {
        while (this.isAlive()) {
          try {
            this.join();
          } catch (final InterruptedException e) {
            JobExecutorThread.LOG.trace("joining " + this.getName() + " got interrupted");
          }
        }
      }
    } else {
      JobExecutorThread.LOG.trace("ignoring deactivate: " + this.getName() + " is not active");
    }
  }
}
