/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.jobexecutor.pvm;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.internal.jobexecutor.AbstractJobExecutor;
import org.ow2.orchestra.pvm.internal.jobexecutor.AcquireJobsCmd;
import org.ow2.orchestra.pvm.internal.jobexecutor.GetNextDueDateCmd;
import org.ow2.orchestra.pvm.internal.jobexecutor.JobExecutor;
import org.ow2.orchestra.pvm.internal.jobexecutor.JobHistoryEntry;

/**
 * manager for jobImpl execution threads and their configuration.
 *
 * @author Tom Baeyens, Guillaume Porcher
 */
public class PvmJobExecutor extends AbstractJobExecutor implements JobExecutor {

  private int nbrOfThreads = 3;
  private int historySize = 200;
  private JobExecutorThreadPool threadPool;

  private DispatcherThread dispatcherThread = null;

  /**
   * queue to dispatch collections of jobDbids to the JobExecutorThreads, which
   * are competing readers.
   */
  private BlockingQueue<Collection<Long>> jobDbidsQueue = null;

  private final List<JobHistoryEntry> history = new ArrayList<JobHistoryEntry>();

  /* (non-Javadoc)
   * @see org.ow2.orchestra.pvm.internal.jobexecutor.IJobExecutor#start()
   */
  @Override
  public synchronized void start() {
    if (this.commandService == null) {
      throw new PvmException(
          "no command executor available in jobImpl executor");
    }
    if (!this.isActive) {
      this.acquireJobsCommand = new AcquireJobsCmd(this.getName(), this.getLockMillis());
      this.nextDueDateCommand = new GetNextDueDateCmd();

      // the max capacity of the jobDbidsQueue is set to
      // nbrOfJobExecutorThreads.
      // That way, the distpatcher thread will be stalled if enough jobs are
      // acquired.
      this.jobDbidsQueue = new ArrayBlockingQueue<Collection<Long>>(this.nbrOfThreads,
          true);

      this.isActive = true;
      AbstractJobExecutor.LOG.trace("starting jobImpl executor threads for jobImpl executor '"
          + this.name + "'...");
      this.threadPool = JobExecutorThreadPoolFactory.getNewInstance(this);
      this.threadPool.start();

      AbstractJobExecutor.LOG.trace("starting dispatcher thread for jobImpl executor '" + this.name
          + "'...");
      this.dispatcherThread = new DispatcherThread(this);
      this.dispatcherThread.start();

    } else {
      AbstractJobExecutor.LOG.trace("ignoring start: jobImpl executor '" + this.name
          + "' is already started'");
    }
  }

  /* (non-Javadoc)
   * @see org.ow2.orchestra.pvm.internal.jobexecutor.IJobExecutor#stop()
   */
  @Override
  public synchronized void stop() {
    this.stop(false);
  }

  /* (non-Javadoc)
   * @see org.ow2.orchestra.pvm.internal.jobexecutor.IJobExecutor#stop(boolean)
   */
  @Override
  public synchronized void stop(final boolean join) {
    AbstractJobExecutor.LOG.debug("stopping jobImpl executor");
    if (this.isActive) {
      this.isActive = false;
      this.dispatcherThread.deactivate(true);
      this.waitTillQueueEmpty();
      this.threadPool.stop(join);
    } else {
      AbstractJobExecutor.LOG.trace("ignoring stop: jobImpl executor '" + this.name + "' not started");
    }
  }

  protected void waitTillQueueEmpty() {
    while (!this.jobDbidsQueue.isEmpty()) {
      AbstractJobExecutor.LOG.trace("waiting for jobImpl-id-queue to become empty");
      try {
        Thread.sleep(200);
      } catch (final InterruptedException e) {
        AbstractJobExecutor.LOG
            .trace("waiting for jobImpl-id-queue to become empty got interrupted");
      }
    }
  }

  protected BlockingQueue<Collection<Long>> getJobDbidsQueue() {
    return this.jobDbidsQueue;
  }

  // getters //////////////////////////////////////////////////////////////////

  public int getHistorySize() {
    return this.historySize;
  }

  public int getNbrOfThreads() {
    return this.nbrOfThreads;
  }

  /* (non-Javadoc)
   * @see org.ow2.orchestra.pvm.internal.jobexecutor.IJobExecutor#getDispatcherThread()
   */
  @Override
  public DispatcherThread getDispatcherThread() {
    return this.dispatcherThread;
  }

  public List<JobHistoryEntry> getHistory() {
    return this.history;
  }

  // configuration setters ////////////////////////////////////////////////////

  public void setNbrOfJobExecutorThreads(final int nbrOfJobExecutorThreads) {
    this.nbrOfThreads = nbrOfJobExecutorThreads;
  }

  public void setHistoryMaxSize(final int historyMaxSize) {
    this.historySize = historyMaxSize;
  }
}
