/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.model;

import java.util.List;

import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.model.Condition;
import org.ow2.orchestra.pvm.model.Event;
import org.ow2.orchestra.pvm.model.Transition;

/**
 * @author Tom Baeyens
 */
public class TransitionImpl extends ObservableElementImpl implements Transition {

  private static final long serialVersionUID = 1L;

  protected NodeImpl source;
  protected NodeImpl destination;
  protected Descriptor conditionDescriptor;
  protected Descriptor waitConditionDescriptor;
  protected boolean isTakeAsync;

  /* Use one of the NodeImpl.createOutgoingTransition methods instead. */
  TransitionImpl() {
    super();
  }

  public void makeDefault() {
    this.source.setDefaultTransition(this);
  }

  @Override
  public String toString() {
    final String destinationName = this.destination != null ? this.destination.getName() : null;
    final String sourceName = this.source != null ? this.source.getName() : null;
    return (sourceName != null ? "(" + sourceName + ")--" : "--")
        + (this.name != null ? this.name + "-->" : ">")
        + (destinationName != null ? "(" + destinationName + ")" : "");
  }

  public EventImpl createEvent() {
    final EventImpl event = new EventImpl();
    event.setName(Event.TRANSITION_TAKE);
    this.addEvent(event);
    return event;
  }

  // get parent ///////////////////////////////////////////////////////////////

  /**
   * the first common parent between the source and the destination node. The
   * source and destination itself are included in the search except if source
   * and destination are equal. In that case (self-transition), then it's the
   * parent of the node.
   */
  @Override
  public ObservableElementImpl getParent() {
    // if one of the two ends is null
    if ((this.source == null) || (this.destination == null)) {
      // the process definition is returned
      return this.processDefinition;
    }
    // if this is a self transition
    if (this.source.equals(this.destination)) {
      // the parent of the node is returned
      return this.source.getParent();
    }
    // if none of the above, we search for the first common element in the
    // parent chains
    final List<ObservableElementImpl> sourceChain = this.source.getParentChain();
    final List<ObservableElementImpl> destinationChain = this.destination.getParentChain();
    for (final ObservableElementImpl sourceElement : sourceChain) {
      for (final ObservableElementImpl destinationElement : destinationChain) {
        if (sourceElement.equals(destinationElement)) {
          return sourceElement;
        }
      }
    }
    return null;
  }

  // getters and setters //////////////////////////////////////////////////////

  @Override
  public void setName(final String name) {
    // if there is no source node associated with this transition
    if (this.source == null) {
      // it s just a setter
      this.name = name;

    } else { // otherwise
      // make sure the source node's leavingTransitionsMap remains up to date
      if (this.name != null) {
        this.source.removeOutgoingTransition(this);
      }
      this.name = name;
      if (name != null) {
        this.source.addOutgoingTransition(this);
      }
    }
  }

  public Condition getCondition() {
    if (this.conditionDescriptor == null) {
      return null;
    }
    return (Condition) WireContext.create(this.conditionDescriptor);
  }

  public Condition getWaitCondition() {
    if (this.waitConditionDescriptor == null) {
      return null;
    }
    return (Condition) WireContext.create(this.waitConditionDescriptor);
  }

  public EventImpl getEvent() {
    return this.getEvent(Event.TRANSITION_TAKE);
  }

  public NodeImpl getSource() {
    return this.source;
  }

  public void setSource(final NodeImpl source) {
    this.source = source;
  }

  public NodeImpl getDestination() {
    return this.destination;
  }

  public void setDestination(final NodeImpl destination) {
    this.destination = destination;
  }

  /** see <a href="#conditions">Conditions</a> */
  public Descriptor getConditionDescriptor() {
    return this.conditionDescriptor;
  }

  /** see <a href="#conditions">Conditions</a> */
  public void setConditionDescriptor(final Descriptor conditionDescriptor) {
    this.conditionDescriptor = conditionDescriptor;
  }

  /** see <a href="#waitconditions">Wait conditions</a> */
  public Descriptor getWaitConditionDescriptor() {
    return this.waitConditionDescriptor;
  }

  /** see <a href="#waitconditions">Wait conditions</a> */
  public void setWaitConditionDescriptor(final Descriptor waitConditionDescriptor) {
    this.waitConditionDescriptor = waitConditionDescriptor;
  }

  public boolean isTakeAsync() {
    return this.isTakeAsync;
  }

  public void setTakeAsync(final boolean isTakeAsync) {
    this.isTakeAsync = isTakeAsync;
  }
}
