/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.svc;

import java.util.Random;

import org.hibernate.StaleStateException;
import org.hibernate.exception.GenericJDBCException;
import org.hibernate.exception.LockAcquisitionException;
import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.log.Log;

/**
 * retries the command execution in case hibernate throws optimistic locking
 * (StaleObjectException) exceptions.
 *
 * @author Tom Baeyens
 */
public class RetryInterceptor extends Interceptor {

  private static final Log LOG = Log.getLog(RetryInterceptor.class.getName());

  private static final Random RANDOM = new Random();
  private int retries = 3;
  private long delay = 50;
  private int delayFactor = 2;

  public <T> T execute(final Command<T> command) {
    int attempt = 1;
    long sleepTime = this.delay;
    long attemptFactor = this.delayFactor;
    while (attempt <= this.retries) {
      if (attempt > 1) {
        RetryInterceptor.LOG.trace("retrying...");
      }
      try {

        return this.next.execute(command);

      } catch (final RuntimeException e) {
        if (this.shouldRetry(e)) {
          RetryInterceptor.LOG.trace("command failed: " + e);
          attempt++;
          if (attempt > this.retries) {
            RetryInterceptor.LOG.error("gave up after " + attempt + " attempts", e);
            throw e;
          }
          RetryInterceptor.LOG.trace("waiting " + sleepTime + " millis");
          try {
            Thread.sleep(sleepTime);
          } catch (final InterruptedException e1) {
            RetryInterceptor.LOG.trace("retry sleeping got interrupted");
          }
          attemptFactor = attemptFactor * this.delayFactor;
          sleepTime = this.delay * RetryInterceptor.RANDOM.nextInt(this.delayFactor);
        } else {
          throw e;
        }
      }
    }
    throw new PvmException("gave up after " + attempt + " attempts");
  }

  protected boolean shouldRetry(final RuntimeException e) {
    if (e instanceof StaleStateException
        || e instanceof LockAcquisitionException
        || e instanceof GenericJDBCException) {
      return true;
    }
    return false;
  }

  public int getRetries() {
    return this.retries;
  }

  public void setRetries(final int retries) {
    this.retries = retries;
  }

  public long getDelay() {
    return this.delay;
  }

  public void setDelay(final long delay) {
    this.delay = delay;
  }

  public long getDelayFactor() {
    return this.delayFactor;
  }

  public void setDelayFactor(final int delayFactor) {
    this.delayFactor = delayFactor;
  }
}
