/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.type;

import java.io.Serializable;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.internal.model.ExecutionImpl;

/**
 * is a orchestra-internal class that serves as a base class for classes that store
 * variable values in the database.
 */
public abstract class Variable implements Serializable {

  private static final long serialVersionUID = 1L;

  protected long dbid = -1;
  protected int dbversion = 0;

  protected String key = null;
  protected Converter converter = null;
  protected String queryText = null;
  protected ExecutionImpl processInstance;

  // constructors /////////////////////////////////////////////////////////////

  // abstract methods /////////////////////////////////////////////////////////

  /**
   * is true if this variable-instance supports the given value, false
   * otherwise.
   */
  public abstract boolean isStorable(Object value);

  /**
   * is the value, stored by this variable instance.
   */
  protected abstract Object getObject();

  /**
   * stores the value in this variable instance.
   */
  protected abstract void setObject(Object value);

  // variable management //////////////////////////////////////////////////////

  public boolean supports(final Object value) {
    if (this.converter != null) {
      return this.converter.supports(value);
    }
    return this.isStorable(value);
  }

  public void setValue(Object value) {
    if (this.converter != null) {
      if (!this.converter.supports(value)) {
        throw new PvmException(
            "the converter '"
                + this.converter.getClass().getName()
                + "' in variable instance '"
                + this.getClass().getName()
                + "' does not support values of type '"
                + value.getClass().getName()
                + "'.  to change the type of a variable, you have to delete it first");
      }
      value = this.converter.convert(value);
    }
    if ((value != null) && (!this.isStorable(value))) {
      throw new PvmException("variable instance '" + this.getClass().getName()
          + "' does not support values of type '" + value.getClass().getName()
          + "'.  to change the type of a variable, you have to delete it first");
    }
    this.setObject(value);
  }

  public Object getValue() {
    Object value = this.getObject();
    if ((value != null) && (this.converter != null)) {
      value = this.converter.revert(value);
    }
    return value;
  }

  // utility methods /////////////////////////////////////////////////////////

  @Override
  public String toString() {
    return "${" + this.key + "}";
  }

  public Type getType() {
    final Type type = new Type();
    type.setConverter(this.converter);
    type.setVariableClass(this.getClass());
    return type;
  }

  // getters and setters //////////////////////////////////////////////////////

  public String getKey() {
    return this.key;
  }

  public Converter getConverter() {
    return this.converter;
  }

  public void setConverter(final Converter converter) {
    this.converter = converter;
  }

  public void setKey(final String key) {
    this.key = key;
  }

  public ExecutionImpl getProcessInstance() {
    return this.processInstance;
  }

  public void setProcessInstance(final ExecutionImpl processInstance) {
    this.processInstance = processInstance;
  }
}
