/**
 * Copyright (C) 2009  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.ow2.orchestra.pvm.internal.util;

import org.ow2.orchestra.pvm.internal.xml.Parse;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Element;


/**
 * @author Guillaume Porcher
 *
 */
public abstract class PvmXmlUtil {

  private PvmXmlUtil() { }

  /**
   * convenience method to combine extraction of a string attribute value.
   *
   * If the attribute exists, it is returned. If the attribute is not present,
   * the defaultValue is returned. The attribute is not present and it is
   * required, a problem will be added to the parse.
   */
  public static String attribute(final Element element, final String attributeName,
      final boolean required, final Parse parse) {
    if (element.hasAttribute(attributeName)) {
      final String value = element.getAttribute(attributeName);
      if (required && "".equals(value)) {
        parse.addProblem("attribute <" + XmlUtil.getTagLocalName(element) + " "
            + attributeName + "=\"\" is empty");
      }
      return value;
    }

    if (required) {
      parse.addProblem("attribute <" + XmlUtil.getTagLocalName(element) + " "
          + attributeName + "=\"...\" is required");
    }

    return null;
  }

  /** parse an attribute as an integer. */
  public static Integer attributeInteger(final Element element, final String attributeName,
      final boolean required, final Parse parse) {
    final String valueText = PvmXmlUtil.attribute(element, attributeName, required, parse);

    if (valueText != null) {
      try {
        return Integer.parseInt(valueText);
      } catch (final NumberFormatException e) {
        parse.addProblem(XmlUtil.errorMessageAttribute(element, attributeName,
            valueText, "value not parsable as integer"));
      }
    }

    return null;
  }

  /** parse an attribute as an boolean. */
  public static Boolean attributeBoolean(final Element element, final String attributeName,
      final boolean required, final Parse parse) {
    return PvmXmlUtil.attributeBoolean(element, attributeName, required, parse, null);
  }

  /** parse an attribute as an boolean. */
  public static Boolean attributeBoolean(final Element element, final String attributeName,
      final boolean required, final Parse parse, final Boolean defaultValue) {
    final String valueText = PvmXmlUtil.attribute(element, attributeName, required, parse);
    if (valueText != null) {
      final Boolean value = XmlUtil.parseBooleanValue(valueText);
      if (value == null) {
        parse
            .addProblem(XmlUtil.errorMessageAttribute(element, attributeName,
                valueText,
                "value not in {true, enabled, on, false, disabled, off}"));
      }
      return value;
    }
    return defaultValue;
  }

}
