package org.ow2.orchestra.pvm.internal.wire.operation;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.util.FilterListener;
import org.ow2.orchestra.pvm.internal.util.Listener;
import org.ow2.orchestra.pvm.internal.util.Observable;
import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireDefinition;
import org.ow2.orchestra.pvm.internal.wire.WireException;
import org.ow2.orchestra.pvm.internal.wire.descriptor.ArgDescriptor;

/**
 * subscribes to an {@link Observable observable}.
 *
 * <p>
 * The target object can be a {@link Listener} or a specific method to call can
 * be specified (by {@link #setMethodName(String)})
 * </p>
 *
 * <p>
 * The event can be filtered by specifying a {@link org.ow2.orchestra.pvm.env.Context} (with
 * {@link #setContextName(String)}), objects to observe (with
 * {@link #setObjectNames(List)}) and events to observe (with
 * {@link #setEventNames(List)}). If the objects or events are not specified,
 * then all objects and events are observed.
 * </p>
 *
 * <p>
 * The {@link #setWireEvents(boolean)} specifies if the object or the
 * {@link Descriptor} events should be observed.
 * </p>
 *
 * @author Tom Baeyens
 * @author Guillaume Porcher (documentation)
 *
 */
public class SubscribeOperation implements Operation {

  private static final long serialVersionUID = 1L;
  private static Log log = Log.getLog(SubscribeOperation.class.getName());

  private String contextName = null;
  private List<String> eventNames = null;

  private boolean wireEvents = false;
  private List<String> objectNames = null;
  private String methodName = null;
  private List<ArgDescriptor> argDescriptors = null;

  public void apply(final Object target, final WireContext targetWireContext) {
    Listener listener = null;

    // if a method has to be invoked, rather then using the observable interface
    if (this.methodName != null) {
      listener = new MethodInvokerListener(this.methodName, this.argDescriptors,
          targetWireContext, target);
    } else {
      try {
        listener = (Listener) target;
      } catch (final ClassCastException e) {
        throw new WireException("couldn't subscribe object " + target
            + " because it is not a Listener");
      }
    }

    // if there is a filter specified on the event names
    if ((this.eventNames != null) && (!this.eventNames.isEmpty())) {
      listener = new FilterListener(listener, this.eventNames);
    }

    // identify the wireContext
    WireContext wireContext = null;
    if (this.contextName != null) {
      final Environment environment = Environment.getCurrent();
      if (environment != null) {
        try {
          wireContext = (WireContext) environment.getContext(this.contextName);
          if (wireContext == null) {
            throw new WireException("couldn't subscribe because context "
                + this.contextName + " doesn't exist");
          }
        } catch (final ClassCastException e) {
          throw new WireException("couldn't subscribe because context "
              + this.contextName + " is not a WireContext", e);
        }
      } else {
        throw new WireException("couldn't get context " + this.contextName
            + " for subscribe because no environment available in context "
            + targetWireContext);
      }
    } else {
      wireContext = targetWireContext;
    }

    if (this.wireEvents) {
      final WireDefinition wireDefinition = wireContext.getWireDefinition();

      // if there are objectNames specified
      if (this.objectNames != null) {
        // subscribe to the descriptors for the all objectNames
        for (final String objectName : this.objectNames) {
          final Descriptor descriptor = wireDefinition.getDescriptor(objectName);
          this.subscribe(listener, descriptor);
        }

        // if no objectNames are specified, subscribe to all the descriptors
      } else {
        final Set<Descriptor> descriptors = new HashSet<Descriptor>(wireDefinition
            .getDescriptors().values());
        for (final Descriptor descriptor : descriptors) {
          this.subscribe(listener, descriptor);
        }
      }

    } else if ((this.objectNames != null) && (!this.objectNames.isEmpty())) {
      // for every objectName
      for (final String objectName : this.objectNames) {
        // subscribe to the objects themselves
        final Object object = wireContext.get(objectName);
        if (object == null) {
          throw new WireException("couldn't subscribe to object in context "
              + wireContext.getName() + ": object " + objectName
              + " unavailable");
        }
        if (!(object instanceof Observable)) {
          throw new WireException("couldn't subscribe to object in context "
              + wireContext.getName() + ": object " + objectName + " ("
              + object.getClass().getName() + ") isn't "
              + Observable.class.getName());
        }
        this.subscribe(listener, (Observable) object);
      }

    } else {
      // subscribe to the context
      this.subscribe(listener, wireContext);
    }
  }

  void subscribe(final Listener listener, final Observable observable) {
    SubscribeOperation.log.trace("adding " + listener + " as listener to " + observable);
    observable.addListener(listener);
  }

  /**
   * Gets the list of argDescriptor used to create the arguments given to the
   * method (only if a specific method has to be called).
   */
  public List<ArgDescriptor> getArgDescriptors() {
    return this.argDescriptors;
  }

  /**
   * Sets the list of argDescriptor used to create the arguments given to the
   * method.
   */
  public void setArgDescriptors(final List<ArgDescriptor> argDescriptors) {
    this.argDescriptors = argDescriptors;
  }

  /**
   * Gets the list of events to listen to.
   */
  public List<String> getEventNames() {
    return this.eventNames;
  }

  /**
   * Sets the list of events to listen to.
   */
  public void setEventNames(final List<String> eventNames) {
    this.eventNames = eventNames;
  }

  /**
   * Gets the name of the method to invoke when an event is received.
   */
  public String getMethodName() {
    return this.methodName;
  }

  /**
   * Sets the name of the method to invoke when an event is received.
   */
  public void setMethodName(final String methodName) {
    this.methodName = methodName;
  }

  /**
   * Gets the name of the WireContext where the Observable should be found.
   */
  public String getContextName() {
    return this.contextName;
  }

  /**
   * Sets the name of the WireContext where the Observable should be found.
   */
  public void setContextName(final String contextName) {
    this.contextName = contextName;
  }

  /**
   * Gets the list of name of the Observable objects to observe.
   */
  public List<String> getObjectNames() {
    return this.objectNames;
  }

  /**
   * Sets the list of name of the Observable objects to observe.
   */
  public void setObjectNames(final List<String> objectNames) {
    this.objectNames = objectNames;
  }

  /**
   * <p>
   * <code>true</code> if the target object will listen to Descriptor related
   * events.
   * </p>
   * <p>
   * <code>false</code> if the target object will listen to the object instance
   * events.
   * </p>
   */
  public boolean isWireEvents() {
    return this.wireEvents;
  }

  /**
   * Sets if the object should listen to descriptor events or to events fired by
   * the named object.
   * <p>
   * <code>true</code> if the target object will listen to Descriptor related
   * events.
   * </p>
   * <p>
   * <code>false</code> if the target object will listen to the object instance
   * events.
   * </p>
   */
  public void setWireEvents(final boolean wireEvents) {
    this.wireEvents = wireEvents;
  }
}
