/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.ow2.orchestra.pvm.internal.stream.ByteArrayStreamSource;
import org.ow2.orchestra.pvm.internal.stream.FileStreamSource;
import org.ow2.orchestra.pvm.internal.stream.InputStreamSource;
import org.ow2.orchestra.pvm.internal.stream.ResourceStreamSource;
import org.ow2.orchestra.pvm.internal.stream.StreamSource;
import org.ow2.orchestra.pvm.internal.stream.UrlStreamSource;
import org.ow2.orchestra.pvm.internal.util.IoUtil;
import org.ow2.orchestra.pvm.internal.xml.Parser;
import org.w3c.dom.Document;

/**
 * a deployment unit, containing all information to create a process definition
 * that will be deployed in the persistent store of the Process Virtual Machine.
 *
 * @author Tom Baeyens
 */
public class Deployment implements Serializable {

  private static final long serialVersionUID = 1L;

  private static final Parser PARSER = new Parser();

  protected String name;
  protected String language;

  protected Map<String, StreamSource> files;
  protected Map<String, Document> documents;

  protected ProcessDefinition processDefinition;

  public Deployment() {
  }

  public Deployment(final ProcessDefinition processDefinition) {
    this.setProcessDefinition(processDefinition);
  }

  public String getLanguage() {
    return this.language;
  }

  public void setLanguage(final String language) {
    this.language = language;
  }

  public void addResource(final String resource) {
    this.addStreamSource(resource, new ResourceStreamSource(resource));
  }

  public void addFile(final File file) {
    this.addStreamSource(file.getAbsolutePath(), new FileStreamSource(file));
  }

  public void addUrl(final URL url) {
    this.addStreamSource(url.toString(), new UrlStreamSource(url));
  }

  public void addInputStream(final String name, final InputStream inputStream) {
    this.addStreamSource(name, new InputStreamSource(inputStream));
  }

  public void addArchiveResource(final String resource) {
    this.name = resource;
    final ResourceStreamSource streamSource = new ResourceStreamSource(resource);
    this.addStreamSource(resource, streamSource);
  }

  public void addArchiveFile(final File file) {
    this.addStreamSource(file.getAbsolutePath(), new FileStreamSource(file));
  }

  public void addArchiveUrl(final URL url) {
    this.addStreamSource(url.toString(), new UrlStreamSource(url));
  }

  public void addArchive(final ZipInputStream zipInputStream) {
    try {
      ZipEntry zipEntry = zipInputStream.getNextEntry();
      while (zipEntry != null) {
        final String entryName = zipEntry.getName();
        final byte[] bytes = IoUtil.readBytes(zipInputStream);
        if (bytes != null) {
          this.addStreamSource(entryName, new ByteArrayStreamSource(bytes));
        }
        zipEntry = zipInputStream.getNextEntry();
      }
    } catch (final Exception e) {
      throw new PvmException("couldn't read zip archive", e);
    }
  }

  /** recursively adds all files in a directory using the relative file names */
  public void addDirectory(final String directory) {
    if (directory == null) {
      throw new PvmException("directory is null");
    }
    this.addDirectory(new File(directory), "", false);
  }

  /** recursively adds all files in a directory using the canonical file names */
  public void addDirectoryCanonical(final String directory) {
    if (directory == null) {
      throw new PvmException("directory is null");
    }
    this.addDirectory(new File(directory), "", true);
  }

  /** recursively adds all files in a directory using the relative file names */
  public void addDirectory(final File directory) {
    this.addDirectory(directory, "", false);
  }

  /** recursively adds all files in a directory using the canonical file names */
  public void addDirectoryCanonical(final File directory) {
    this.addDirectory(directory, "", true);
  }

  protected void addDirectory(final File directory, final String relativeDirectoryName,
      final boolean canonicalPathNames) {
    if (directory == null) {
      throw new PvmException("directory is null");
    }
    if (!directory.isDirectory()) {
      throw new PvmException(directory.getAbsolutePath()
          + " is not a directory");
    }

    final File[] files = directory.listFiles();
    if (files != null) {
      for (final File file : files) {
        final String relativeFileName = canonicalPathNames ? null : relativeDirectoryName + "/" + file.getName();
        if (file.isFile()) {
          if (canonicalPathNames) {
            try {
              this.addStreamSource(file.getCanonicalPath(), new FileStreamSource(
                  file));
            } catch (final IOException e) {
              throw new PvmException("can't get canonical path name for "
                  + file);
            }
          } else {
            this.addStreamSource(relativeFileName, new FileStreamSource(file));
          }
        } else if (file.isDirectory()) {
          this.addDirectory(file, relativeFileName, canonicalPathNames);
        }
      }
    }
  }

  protected void addStreamSource(final String name, final StreamSource streamSource) {
    if (this.name == null) {
      this.name = name;
    }
    if (this.files == null) {
      this.files = new HashMap<String, StreamSource>();
    }
    this.files.put(name, streamSource);
  }

  public InputStream getFile(final String name) {
    if (this.files == null) {
      return null;
    }
    final StreamSource streamSource = this.files.get(name);
    return (streamSource != null ? streamSource.openStream() : null);
  }

  public Set<String> getFileNames() {
    if (this.files == null) {
      return Collections.EMPTY_SET;
    }
    return this.files.keySet();
  }

  public Document getDocument(final String name) {
    if ((this.documents != null) && (this.documents.containsKey(name))) {
      return this.documents.get(name);
    }
    if ((this.files != null) && (this.files.containsKey(name))) {
      final InputStream fileStream = this.getFile(name);
      return Deployment.PARSER.createParse().setInputStream(fileStream).execute()
          .checkProblems("deployment file " + name).getDocument();
    }
    return null;
  }

  public ProcessDefinition getProcessDefinition() {
    return this.processDefinition;
  }

  public void setProcessDefinition(final ProcessDefinition processDefinition) {
    if (this.language == null) {
      this.language = "api";
    }
    this.processDefinition = processDefinition;
  }

  public String getName() {
    return this.name;
  }

  public void setName(final String name) {
    this.name = name;
  }
}
