/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.cal;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Date;

/**
 * is part of a day that can for example be used to represent business hours.
 *
 */
public class DayPart implements Serializable {

  private static final long serialVersionUID = 1L;

  protected long oid = -1;
  protected int version = 0;
  protected int fromHour = -1;
  protected int fromMinute = -1;
  protected int toHour = -1;
  protected int toMinute = -1;
  protected Day day = null;
  protected int index = -1;

  public Date add(final Date date, final long millis, final boolean isBusinessTime) {
    Date end = null;

    final BusinessCalendar businessCalendar = this.day.getBusinessCalendar();
    final Calendar calendar = businessCalendar.createCalendar();
    calendar.setTime(date);
    final int hour = calendar.get(Calendar.HOUR_OF_DAY);
    final int minute = calendar.get(Calendar.MINUTE);

    final long dateMilliseconds = ((hour * 60L) + minute) * 60 * 1000;
    final long dayPartEndMilleseconds = ((this.toHour * 60L) + this.toMinute) * 60 * 1000;
    final long millisecondsInThisDayPart = dayPartEndMilleseconds - dateMilliseconds;

    if (millis <= millisecondsInThisDayPart) {
      end = new Date(date.getTime() + millis);
    } else {
      final long remainderMillis = millis - millisecondsInThisDayPart;
      final Date dayPartEndDate = new Date(date.getTime() + millis - remainderMillis);

      final Object[] result = new Object[2];
      this.day.findNextDayPartStart(this.index + 1, dayPartEndDate, result);
      final Date nextDayPartStart = (Date) result[0];
      final DayPart nextDayPart = (DayPart) result[1];

      end = nextDayPart.add(nextDayPartStart, remainderMillis, isBusinessTime);
    }

    return end;
  }

  public boolean isStartAfter(final Date date) {
    final Calendar calendar = this.day.getBusinessCalendar().createCalendar();
    calendar.setTime(date);
    final int hour = calendar.get(Calendar.HOUR_OF_DAY);
    final int minute = calendar.get(Calendar.MINUTE);

    return (hour < this.fromHour) || ((hour == this.fromHour) && (minute <= this.fromMinute));
  }

  public boolean includes(final Date date) {
    final Calendar calendar = this.day.getBusinessCalendar().createCalendar();
    calendar.setTime(date);
    final int hour = calendar.get(Calendar.HOUR_OF_DAY);
    final int minute = calendar.get(Calendar.MINUTE);

    return ((this.fromHour < hour)
        || ((this.fromHour == hour) && (this.fromMinute <= minute))) && ((hour < this.toHour)
        || ((hour == this.toHour) && (minute <= this.toMinute)));
  }

  public Date getStartTime(final Date date) {
    final Calendar calendar = this.day.getBusinessCalendar().createCalendar();
    calendar.setTime(date);
    calendar.set(Calendar.HOUR_OF_DAY, this.fromHour);
    calendar.set(Calendar.MINUTE, this.fromMinute);
    return calendar.getTime();
  }

  public Day getDay() {
    return this.day;
  }

  public void setDay(final Day day) {
    this.day = day;
  }

  public int getFromHour() {
    return this.fromHour;
  }

  public void setFromHour(final int fromHour) {
    this.fromHour = fromHour;
  }

  public int getFromMinute() {
    return this.fromMinute;
  }

  public void setFromMinute(final int fromMinute) {
    this.fromMinute = fromMinute;
  }

  public int getIndex() {
    return this.index;
  }

  public void setIndex(final int index) {
    this.index = index;
  }

  public int getToHour() {
    return this.toHour;
  }

  public void setToHour(final int toHour) {
    this.toHour = toHour;
  }

  public int getToMinute() {
    return this.toMinute;
  }

  public void setToMinute(final int toMinute) {
    this.toMinute = toMinute;
  }
}
