/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.cal;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ow2.orchestra.pvm.PvmException;

/**
 * represents a time duration.
 *
 * <p>
 * With the constructor {link {@link #Duration(String)} you can create a
 * Duration from a text representation. The syntax is as follows
 * </p>
 *
 * <pre>
 * duration = part [',' part | 'and' part]*
 * part = number ['business'] unit
 * number = (0..9)+
 * unit = (y|year|years|month|months|w|week|weeks|d|day|days|h|hour|hours|min|minute|minutes|s|sec|second|seconds|milli|millis|millisecond|milliseconds)
 * </pre>
 *
 * <p>
 * Duration is immutable.
 * </p>
 */
public class Duration implements Serializable {

  private static final long serialVersionUID = 1L;

  protected boolean isBusinessTime;
  protected int millis;
  protected int seconds;
  protected int minutes;
  protected int hours;
  protected int days;
  protected int weeks;
  protected int months;
  protected int years;

  /** constructor for persistence. note that this type is to be immutable. */
  protected Duration() {
  }

  /**
   * parses the duration from a text
   *
   * duration = part [',' part | 'and' part]* part = number ['business'] unit
   * number = (0..9)+ unit =
   * (y|year|years|month|months|w|week|weeks|d|day|days|h
   * |hour|hours|min|minute|minutes
   * |s|sec|second|seconds|milli|millis|millisecond|milliseconds)
   *
   * @throws PvmException
   *           if the parsing is unsuccessful
   */
  public Duration(final String text) {
    if (text == null) {
      throw new PvmException("text is null");
    }

    for (final String part : this.splitInParts(text)) {
      this.parsePart(part);
    }

    this.isBusinessTime = text.indexOf("business") != -1;
  }

  public Duration(final boolean isBusinessTime, final int millis, final int seconds, final int minutes,
      final int hours, final int days, final int weeks, final int months, final int years) {
    this.isBusinessTime = isBusinessTime;
    this.millis = millis;
    this.seconds = seconds;
    this.minutes = minutes;
    this.hours = hours;
    this.days = days;
    this.weeks = weeks;
    this.months = months;
    this.years = years;
  }

  private List<String> splitInParts(String text) {
    final List<String> parts = new ArrayList<String>(2);
    while (text != null) {
      final int commaIndex = text.indexOf(',');
      final int andIndex = text.indexOf(" and ");
      if (((commaIndex == -1) && (andIndex != -1))
          || (((commaIndex != -1) && (andIndex != -1)) && (andIndex < commaIndex))) {
        final String part = text.substring(0, andIndex).trim();
        parts.add(part);
        text = text.substring(andIndex + 5);

      } else if (((commaIndex != -1) && (andIndex == -1))
          || (((commaIndex != -1) && (andIndex != -1)) && (andIndex > commaIndex))) {
        final String part = text.substring(0, commaIndex).trim();
        parts.add(part);
        text = text.substring(commaIndex + 1);

      } else {
        parts.add(text.trim());
        text = null;
      }
    }

    return parts;
  }

  private void parsePart(final String part) {
    int spaceIndex = part.indexOf(' ');
    if (spaceIndex == -1) {
      throw new PvmException("couldn't parse duration part " + part);
    }
    final String quantityText = part.substring(0, spaceIndex).trim();
    spaceIndex = part.lastIndexOf(' ');
    final String unitText = part.substring(spaceIndex + 1).trim().toLowerCase();

    int quantity;
    try {
      quantity = Integer.parseInt(quantityText);
    } catch (final NumberFormatException e) {
      throw new PvmException("couldn't parse quantity " + quantityText
          + " in duration text", e);
    }
    final FieldSetter fieldSetter = Duration.fieldSetters.get(unitText);
    if (fieldSetter == null) {
      throw new PvmException("couldn't parse quantity " + quantityText);
    }
    fieldSetter.set(this, quantity);
  }

  private interface FieldSetter {
    void set(Duration duration, int quantity);
  }

  private static class MillisSetter implements FieldSetter {
    public void set(final Duration duration, final int quantity) {
      duration.millis = quantity;
    }
  }

  private static class SecondSetter implements FieldSetter {
    public void set(final Duration duration, final int quantity) {
      duration.seconds = quantity;
    }
  }

  private static class MinuteSetter implements FieldSetter {
    public void set(final Duration duration, final int quantity) {
      duration.minutes = quantity;
    }
  }

  private static class HourSetter implements FieldSetter {
    public void set(final Duration duration, final int quantity) {
      duration.hours = quantity;
    }
  }

  private static class DaySetter implements FieldSetter {
    public void set(final Duration duration, final int quantity) {
      duration.days = quantity;
    }
  }

  private static class WeekSetter implements FieldSetter {
    public void set(final Duration duration, final int quantity) {
      duration.weeks = quantity;
    }
  }

  private static class MonthSetter implements FieldSetter {
    public void set(final Duration duration, final int quantity) {
      duration.months = quantity;
    }
  }

  private static class YearSetter implements FieldSetter {
    public void set(final Duration duration, final int quantity) {
      duration.years = quantity;
    }
  }

  private static final Map<String, FieldSetter> fieldSetters = new HashMap<String, FieldSetter>();
  static {
    FieldSetter fieldSetter = new MillisSetter();
    Duration.fieldSetters.put("milli", fieldSetter);
    Duration.fieldSetters.put("millis", fieldSetter);
    Duration.fieldSetters.put("millisecond", fieldSetter);
    Duration.fieldSetters.put("milliseconds", fieldSetter);

    fieldSetter = new SecondSetter();
    Duration.fieldSetters.put("s", fieldSetter);
    Duration.fieldSetters.put("sec", fieldSetter);
    Duration.fieldSetters.put("second", fieldSetter);
    Duration.fieldSetters.put("seconds", fieldSetter);

    fieldSetter = new MinuteSetter();
    Duration.fieldSetters.put("min", fieldSetter);
    Duration.fieldSetters.put("minute", fieldSetter);
    Duration.fieldSetters.put("minutes", fieldSetter);

    fieldSetter = new HourSetter();
    Duration.fieldSetters.put("h", fieldSetter);
    Duration.fieldSetters.put("hour", fieldSetter);
    Duration.fieldSetters.put("hours", fieldSetter);

    fieldSetter = new DaySetter();
    Duration.fieldSetters.put("d", fieldSetter);
    Duration.fieldSetters.put("day", fieldSetter);
    Duration.fieldSetters.put("days", fieldSetter);

    fieldSetter = new WeekSetter();
    Duration.fieldSetters.put("w", fieldSetter);
    Duration.fieldSetters.put("week", fieldSetter);
    Duration.fieldSetters.put("weeks", fieldSetter);

    fieldSetter = new MonthSetter();
    Duration.fieldSetters.put("month", fieldSetter);
    Duration.fieldSetters.put("months", fieldSetter);

    fieldSetter = new YearSetter();
    Duration.fieldSetters.put("y", fieldSetter);
    Duration.fieldSetters.put("year", fieldSetter);
    Duration.fieldSetters.put("years", fieldSetter);
  }

  public int getDays() {
    return this.days;
  }

  public int getHours() {
    return this.hours;
  }

  public boolean isBusinessTime() {
    return this.isBusinessTime;
  }

  public int getMillis() {
    return this.millis;
  }

  public int getMinutes() {
    return this.minutes;
  }

  public int getMonths() {
    return this.months;
  }

  public int getSeconds() {
    return this.seconds;
  }

  public int getWeeks() {
    return this.weeks;
  }

  public int getYears() {
    return this.years;
  }
}
