/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.hibernate;

import java.util.Collection;
import java.util.List;

import org.hibernate.Query;
import org.ow2.orchestra.pvm.Execution;
import org.ow2.orchestra.pvm.internal.job.JobImpl;
import org.ow2.orchestra.pvm.internal.jobexecutor.JobDbSession;
import org.ow2.orchestra.pvm.internal.model.ExecutionImpl;
import org.ow2.orchestra.pvm.internal.util.Clock;

/**
 * @author Tom Baeyens
 */
public class HibernateJobDbSession extends HibernateDbSession implements
    JobDbSession {

  public Collection<JobImpl< ? >> findFirstAcquirableJob(final int limit) {
    // query definition can be found at the bottom of resource
    // org/ow2/orchestra/pvm/hibernate.job.hbm.xml
    final Query query = this.session.getNamedQuery("findFirstAcquirableJob");
    query.setTimestamp("now", Clock.getCurrentTime());
    query.setMaxResults(limit);
    return query.list();
  }

  public List<JobImpl< ? >> findExclusiveJobs(final Execution processInstance) {
    // query definition can be found at the bottom of resource
    // org/ow2/orchestra/pvm/hibernate.job.hbm.xml
    final Query query = this.session.getNamedQuery("findExclusiveJobs");
    query.setTimestamp("now", Clock.getCurrentTime());
    query.setEntity("processInstance", processInstance);
    return query.list();
  }

  public JobImpl< ? > findFirstDueJob() {
    // query definition can be found at the bottom of resource
    // org/ow2/orchestra/pvm/hibernate.job.hbm.xml
    final Query query = this.session.getNamedQuery("findFirstDueJob");
    query.setMaxResults(1);
    query.setReadOnly(true);
    return (JobImpl< ? >) query.uniqueResult();
  }

  public JobImpl< ? > getJob(final long jobId) {
    return JobImpl.class.cast(this.session.get(JobImpl.class, jobId));
  }

  @Override
  public void save(final Object entity) {
    if (entity instanceof JobImpl< ? >) {
      this.save((JobImpl< ? >) entity);
    } else {
      super.save(entity);
    }
  }

  public void save(final JobImpl< ? > job) {
    if (job.getExecution() != null) {
      job.getExecution().getJobs().add(job);
    }
    super.save(job);
  }


  @Override
  public void delete(final Object entity) {
    if (entity instanceof JobImpl< ? >) {
      this.delete((JobImpl< ? >) entity);
    } else {
      super.delete(entity);
    }
  }

  public void delete(final JobImpl< ? > job) {
    final ExecutionImpl jobExecution = job.getExecution();
    if (jobExecution != null) {
      jobExecution.getJobs().remove(job);
    }
    super.delete(job);
  }
}
