/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.jobexecutor;

import java.net.InetAddress;
import java.util.Collection;
import java.util.Date;

import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.pvm.internal.job.JobImpl;
import org.ow2.orchestra.pvm.internal.jobexecutor.pvm.PvmJobExecutor;
import org.ow2.orchestra.pvm.internal.log.Log;

/**
 * Common part of jobExecutor implementations based on a Dispatcher thread
 *
 * @author Tom Baeyens
 * @author Guillaume Porcher
 *
 */
public abstract class AbstractJobExecutor implements JobExecutor {

  protected static final Log LOG = Log.getLog(PvmJobExecutor.class.getName());

  public abstract void stop(final boolean join);

  public abstract void stop();

  public abstract void start();

  private static final long serialVersionUID = 1L;
  protected CommandService commandService;
  protected String name = "JobExecutor-" + AbstractJobExecutor.getHostName();
  protected int idleMillis = 5000;
  protected int lockMillis = 1000 * 60 * 30;
  protected Command<Collection<Collection<JobImpl<?>>>> acquireJobsCommand;
  protected Command<Date> nextDueDateCommand;
  protected boolean isActive = false;
  protected DeadJobHandler deadJobHandler;

  protected static String getHostName() {
    try {
      return InetAddress.getLocalHost().getHostAddress();
    } catch (final Exception e) {
      return "unknown";
    }
  }

  public void jobWasAdded() {
    if ((this.getDispatcherThread() != null) && (this.getDispatcherThread().isActive())) {
      this.getDispatcherThread().jobWasAdded();
    }
  }

  /**
   *
   */
  public AbstractJobExecutor() {
    super();
  }

  public String getName() {
    return this.name;
  }

  public int getIdleMillis() {
    return this.idleMillis;
  }

  public boolean isActive() {
    return this.isActive;
  }

  public int getLockMillis() {
    return this.lockMillis;
  }

  public CommandService getCommandExecutor() {
    return this.commandService;
  }

  public Command<Collection<Collection<JobImpl<?>>>> getAcquireJobsCommand() {
    return this.acquireJobsCommand;
  }

  public Command<Date> getNextDueDateCommand() {
    return this.nextDueDateCommand;
  }

  public void setCommandExecutor(final CommandService commandExecutor) {
    this.commandService = commandExecutor;
  }

  public void setName(final String name) {
    this.name = name;
  }

  public void setIdleInterval(final int idleInterval) {
    this.idleMillis = idleInterval;
  }

  public void setMaxLockTime(final int maxLockTime) {
    this.lockMillis = maxLockTime;
  }

  public DeadJobHandler getDeadJobHandler() {
    return this.deadJobHandler;
  }

  /**
   * @param deadJobHandler the deadJobHandler to set
   */
  public void setDeadJobHandler(final DeadJobHandler deadJobHandler) {
    this.deadJobHandler = deadJobHandler;
  }

  protected abstract AbstractDispatcherThread getDispatcherThread();

  protected abstract int getAcquireJobSize();
}