/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.jobexecutor;

import java.io.IOException;
import java.io.PrintWriter;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.env.EnvironmentFactory;
import org.ow2.orchestra.pvm.env.PvmEnvironmentFactory;

/**
 * starts the jobImpl executor on init and closes the orchestra configuration upon
 * destroy. The closing of the orchestra configuration will also shut down the
 * jobImpl executor thread pool. <h1>Config parameters</h1>
 * <table>
 * <tr>
 * <th>Name</th>
 * <th>Description</th>
 * <th>Type</th>
 * <th>Default value</th>
 * </tr>
 * <tr>
 * <td>orchestra.configuration.resource</td>
 * <td>resource location of the orchestra.cfg.xml</td>
 * <td>String</td>
 * <td>orchestra.cfg.xml</td>
 * </tr>
 * </table>
 *
 * <p>
 * Configuration example:
 *
 * <pre>
 * &lt;web-app&gt;
 *   ...
 *   &lt;servlet &gt;
 *     &lt;servlet-name&gt;JobExecutorServlet&lt;/servlet-name&gt;
 *     &lt;servlet-class&gt;org.ow2.orchestra.job.executor.JobExecutorServlet&lt;/servlet-class&gt;
 *     &lt;load-on-startup&gt;1&lt;/load-on-startup&gt;
 *   &lt;/servlet&gt;
 *   &lt;servlet-mapping &gt;
 *     &lt;servlet-name&gt;JobExecutorServlet&lt;/servlet-name&gt;
 *     &lt;url-pattern&gt;/jobexecutor&lt;/url-pattern&gt;
 *   &lt;/servlet-mapping&gt;
 *   ...
 * &lt;/web-app&gt;
 * </pre>
 *
 * </p>
 *
 * @author Tom Baeyens
 */
public class JobExecutorServlet extends HttpServlet {

  private static final long serialVersionUID = 1L;

  private JobExecutor jobExecutor;

  @Override
  public void init() throws ServletException {
    final String configurationResource = this.getInitParameter(
        "orchestra.configuration.resource", "orchestra.cfg.xml");
    final EnvironmentFactory environmentFactory = new PvmEnvironmentFactory(
        configurationResource);
    this.jobExecutor = environmentFactory.get(JobExecutor.class);
    if (this.jobExecutor == null) {
      throw new PvmException("no jobImpl executor configured in resource "
          + configurationResource);
    }
    this.jobExecutor.start();
  }

  @Override
  protected void doGet(final HttpServletRequest request, final HttpServletResponse response)
    throws ServletException, IOException {

    final PrintWriter out = response.getWriter();
    out.println("<html>");
    out.println("<body>");
    out.println("<h2>JBoss orchestra JobImpl Executor Servlet</h2><hr />");
    /*
     * TODO Iterator<Thread> iter =
     * jobExecutor.getJobExecutorThreads().values().iterator(); while
     * (iter.hasNext()) { Thread thread = iter.next();
     * out.println("<h4>"+thread.getName()+"</h4>");
     * out.println("<b>isAlive</b>:"+thread.isAlive()); }
     */
    out.println("</body>");
    out.println("</html>");
  }

  String getInitParameter(final String name, final String defaultValue) {
    final String value = this.getInitParameter(name);
    if (value != null) {
      return value;
    }
    return defaultValue;
  }
}
