/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.model;

import javax.transaction.Status;
import javax.transaction.Synchronization;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.env.Transaction;
import org.ow2.orchestra.pvm.internal.cmd.Command;
import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.pvm.session.DbSession;

public class ExceptionHandlerSynchronization implements Synchronization,
    Command<Object> {

  private final ExceptionHandlerImpl exceptionHandler;

  private static final long serialVersionUID = 1L;

  protected ExecutionImpl execution;
  protected ProcessElementImpl processElement;
  protected Exception exception;
  protected CommandService commandService;

  public ExceptionHandlerSynchronization(final ExceptionHandlerImpl exceptionHandler,
      final ExecutionImpl execution, final Exception exception, final ProcessElementImpl processElement,
      final CommandService commandService) {
    this.exceptionHandler = exceptionHandler;
    this.execution = execution;
    this.exception = exception;
    this.commandService = commandService;
    this.processElement = processElement;
  }

  public void afterCompletion(final int status) {
    if (status != Status.STATUS_ROLLEDBACK) {
      ExceptionHandlerImpl.LOG
          .info("WARNING: no rollback after transactional exception handler. did you forget to rollback the transaction ?");
    }
    this.commandService.execute(this);
  }

  public Object execute(final Environment environment) {
    // reload the execution
    final DbSession dbSession = environment.get(DbSession.class);
    if (dbSession == null) {
      throw new PvmException("no " + DbSession.class.getName()
          + " available in the environment for reloading the execution");
    }
    this.execution = dbSession.get(ExecutionImpl.class, this.execution.getDbid());
    this.exceptionHandler.executeHandler(this.execution, this.exception, this.processElement);
    return null;
  }

  public void beforeCompletion() {
  }

  public void register(final Transaction transaction) {
    // registration of the synchronization is delegated to the
    // AfterTxCompletionListener
    // to avoid a dependency on class Synchronization in class
    // ExceptionHandlerImpl
    transaction.registerSynchronization(this);
  }
}
