/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.model.op;

import org.ow2.orchestra.pvm.activity.Activity;
import org.ow2.orchestra.pvm.internal.job.MessageImpl;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.model.ExecutionImpl;
import org.ow2.orchestra.pvm.internal.model.NodeImpl;

public class ExecuteNode implements AtomicOperation {

  private static Log log = Log.getLog(ExecuteNode.class.getName());

  public boolean isAsync(final ExecutionImpl execution) {
    return execution.getNode().isExecutionAsync();
  }

  public void perform(final ExecutionImpl execution) {
    final NodeImpl node = execution.getNode();

    if (ExecuteNode.log.isTraceEnabled()) {
      if (execution.getName() != null) {
        ExecuteNode.log.trace(execution.toString() + " executes " + node);
      } else {
        ExecuteNode.log.trace("executing " + node);
      }
    }

    final Activity activity = node.getBehaviour();

    try {
      execution.setPropagation(ExecutionImpl.Propagation.UNSPECIFIED);

      activity.execute(execution);

    } catch (final Exception e) {
      execution.handleException(node, null, null, e, execution.toString()
          + " couldn't execute " + activity + " for node " + node);
    }

    // if exception has been catched and exception handler has not disabled propagation,
    // then proceed.
    if (execution.getPropagation() == ExecutionImpl.Propagation.UNSPECIFIED) {
      execution.proceed();
    }
  }

  @Override
  public String toString() {
    return "execute(node)";
  }

  public MessageImpl< ? > createAsyncMessage(final ExecutionImpl execution) {
    return new ExecuteNodeMessage(execution);
  }
}
