/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.model.op;

import java.util.Map;

import org.ow2.orchestra.pvm.activity.ExternalActivity;
import org.ow2.orchestra.pvm.internal.job.MessageImpl;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.model.ExecutionImpl;
import org.ow2.orchestra.pvm.internal.model.ExecutionImpl.Propagation;
import org.ow2.orchestra.pvm.internal.model.NodeImpl;

/**
 * @author Tom Baeyens
 */
public class Signal implements AtomicOperation {

  private static final Log LOG = Log.getLog(Signal.class.getName());

  private final String signalName;
  private final Map<String, Object> parameters;
  private final NodeImpl node;

  public Signal(final String signalName, final Map<String, Object> parameters, final NodeImpl node) {
    this.signalName = signalName;
    this.parameters = parameters;
    this.node = node;
  }

  public boolean isAsync(final ExecutionImpl execution) {
    return execution.getNode().isSignalAsync();
  }

  public void perform(final ExecutionImpl execution) {
    if (execution.getName() != null) {
      Signal.LOG.debug(execution.toString() + " signals " + this.node);
    } else {
      Signal.LOG.debug("signalling " + this.node + ", signal=" + this.signalName);
    }

    final ExternalActivity externalActivity = (ExternalActivity) this.node.getBehaviour();


    try {
      execution.setPropagation(Propagation.UNSPECIFIED);

      externalActivity.signal(execution, this.signalName, this.parameters);

    } catch (final Exception e) {
      execution.handleException(this.node, null, null, e, execution.toString()
          + " couldn't signal node " + this.node);
    }

    // if exception has been catched and exception handler has not disabled propagation,
    // then proceed.
    if (execution.getPropagation() == Propagation.UNSPECIFIED) {
      execution.proceed();
    }
  }

  @Override
  public String toString() {
    return "signal(node)";
  }

  public MessageImpl< ? > createAsyncMessage(final ExecutionImpl execution) {
    return new SignalMessage(execution, this.signalName, this.node);
  }
}
