/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.model.op;

import java.util.ArrayList;
import java.util.List;

import org.ow2.orchestra.pvm.internal.job.MessageImpl;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.model.ExecutionImpl;
import org.ow2.orchestra.pvm.internal.model.NodeImpl;
import org.ow2.orchestra.pvm.internal.model.ObservableElementImpl;
import org.ow2.orchestra.pvm.internal.model.TransitionImpl;
import org.ow2.orchestra.pvm.model.Condition;
import org.ow2.orchestra.pvm.model.Event;

public class TakeTransition implements AtomicOperation {

  private static Log log = Log.getLog(TakeTransition.class.getName());

  public boolean isAsync(final ExecutionImpl execution) {
    return execution.getNode().isLeaveAsync()
        || execution.getTransition().isTakeAsync();
  }

  public void perform(final ExecutionImpl execution) {
    final TransitionImpl transition = execution.getTransition();

    if (execution.getName() != null) {
      TakeTransition.log.debug(execution.toString() + " takes " + transition);
    } else {
      TakeTransition.log.debug("taking " + transition);
    }

    final List<NodeImpl> leftNodes = this.getNodesLeft(execution.getNode(), transition
        .getDestination());
    ExecutionImpl propagatingExecution = execution;
    for (final NodeImpl leftNode : leftNodes) {
      propagatingExecution = propagatingExecution.endNode(leftNode);
    }

    propagatingExecution.setNode(null);
    propagatingExecution.fire(Event.TRANSITION_TAKE, transition);

    boolean wait = false;

    final Condition waitCondition = transition.getWaitCondition();
    if (waitCondition != null) {
      wait = waitCondition.evaluate(propagatingExecution);
    }

    if (!wait) {
      propagatingExecution
          .performAtomicOperation(ExecutionImpl.PROCEED_TO_DESTINATION);
    }
  }

  List<NodeImpl> getNodesLeft(final NodeImpl source, final NodeImpl destination) {
    final List<NodeImpl> nodesLeft = new ArrayList<NodeImpl>();

    if (source.equals(destination)) {
      nodesLeft.add(source);
    } else {
      final List<ObservableElementImpl> destinationChain = destination
          .getParentChain();

      if (!destinationChain.contains(source)) {
        NodeImpl sourceNode = source;
        while ((sourceNode != null) && (!destinationChain.contains(sourceNode))) {
          nodesLeft.add(sourceNode);
          sourceNode = sourceNode.getParentNode();
        }
      }
    }

    return nodesLeft;
  }

  @Override
  public String toString() {
    return "take-transition";
  }

  public MessageImpl< ? > createAsyncMessage(final ExecutionImpl execution) {
    return new TakeTransitionMessage(execution);
  }
}
