/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.svc;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.ow2.orchestra.pvm.Execution;
import org.ow2.orchestra.pvm.ExecutionService;
import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.pvm.internal.cmd.FindExecutionCmd;
import org.ow2.orchestra.pvm.internal.cmd.GetVariablesCmd;
import org.ow2.orchestra.pvm.internal.cmd.SetVariablesCmd;
import org.ow2.orchestra.pvm.internal.cmd.SignalCmd;
import org.ow2.orchestra.pvm.internal.cmd.StartExecutionCmd;
import org.ow2.orchestra.pvm.internal.cmd.StartExecutionInLatestCmd;

/**
 * @author Tom Baeyens
 */
public class CommandExecutionService implements ExecutionService {

  protected CommandService commandService;

  public Execution startExecution(String processDefinitionId) {
    return commandService.execute(new StartExecutionCmd(processDefinitionId,
        null, null));
  }

  public Execution startExecution(String processDefinitionId,
      String executionKey) {
    return commandService.execute(new StartExecutionCmd(processDefinitionId,
        null, executionKey));
  }

  public Execution startExecution(String processDefinitionId,
      Map<String, Object> variables) {
    return commandService.execute(new StartExecutionCmd(processDefinitionId,
        variables, null));
  }

  public Execution startExecution(String processDefinitionId,
      Map<String, Object> variables, String executionKey) {
    return commandService.execute(new StartExecutionCmd(processDefinitionId,
        variables, executionKey));
  }

  public Execution startExecutionInLatest(String processDefinitionName) {
    return commandService.execute(new StartExecutionInLatestCmd(
        processDefinitionName, null, null));
  }

  public Execution startExecutionInLatest(String processDefinitionName,
      Map<String, Object> variables) {
    return commandService.execute(new StartExecutionInLatestCmd(
        processDefinitionName, variables, null));
  }

  public Execution startExecutionInLatest(String processDefinitionName,
      String executionKey) {
    return commandService.execute(new StartExecutionInLatestCmd(
        processDefinitionName, null, executionKey));
  }

  public Execution startExecutionInLatest(String processDefinitionName,
      Map<String, Object> variables, String executionKey) {
    return commandService.execute(new StartExecutionInLatestCmd(
        processDefinitionName, variables, executionKey));
  }

  public Execution signalExecution(String executionId) {
    return commandService.execute(new SignalCmd(executionId, null, null));
  }

  public Execution signalExecution(String executionId, String signalName) {
    return commandService.execute(new SignalCmd(executionId, signalName, null));
  }

  public Execution signalExecution(String executionId, String signalName,
      Map<String, Object> parameters) {
    return commandService.execute(new SignalCmd(executionId, signalName,
        parameters));
  }

  public Execution signalExecution(String executionId,
      Map<String, Object> parameters) {
    return commandService.execute(new SignalCmd(executionId, null, parameters));
  }

  public Execution signalExecution(long executionDbid) {
    return commandService.execute(new SignalCmd(executionDbid, null, null));
  }

  public Execution signalExecution(long executionDbid, String signalName) {
    return commandService
        .execute(new SignalCmd(executionDbid, signalName, null));
  }

  public Execution signalExecution(long executionDbid, String signalName,
      Map<String, Object> parameters) {
    return commandService.execute(new SignalCmd(executionDbid, signalName,
        parameters));
  }

  public Execution signalExecution(long executionDbid,
      Map<String, Object> parameters) {
    return commandService
        .execute(new SignalCmd(executionDbid, null, parameters));
  }

  public Execution signalExecutionByKey(String processDefinitionName,
      String executionKey) {
    return commandService.execute(new SignalCmd(processDefinitionName,
        executionKey, null, null));
  }

  public Execution signalExecutionByKey(String processDefinitionName,
      String executionKey, String signalName) {
    return commandService.execute(new SignalCmd(processDefinitionName,
        executionKey, signalName, null));
  }

  public Execution signalExecutionByKey(String processDefinitionName,
      String executionKey, String signalName, Map<String, Object> parameters) {
    return commandService.execute(new SignalCmd(processDefinitionName,
        executionKey, signalName, parameters));
  }

  public Execution signalExecutionByKey(String processDefinitionName,
      String executionKey, Map<String, Object> parameters) {
    return commandService.execute(new SignalCmd(processDefinitionName,
        executionKey, null, parameters));
  }

  public Execution findExecution(String processDefinitionName, String key) {
    return commandService.execute(new FindExecutionCmd(processDefinitionName,
        key));
  }

  public Execution findExecution(String executionId) {
    return commandService.execute(new FindExecutionCmd(executionId));
  }

  public Object getVariable(String executionId, String variableName) {
    List<String> variableNames = new ArrayList<String>();
    variableNames.add(variableName);
    Map<String, Object> variables = commandService.execute(new GetVariablesCmd(
        executionId, variableNames));
    return variables.get(variableName);
  }

  public Map<String, Object> getVariables(String executionId,
      List<String> variableNames) {
    return commandService.execute(new GetVariablesCmd(executionId,
        variableNames));
  }

  public Execution setVariable(String executionId, String name, Object value) {
    SetVariablesCmd cmd = new SetVariablesCmd(executionId);
    cmd.addVariable(name, value);
    return commandService.execute(cmd);
  }

  public Execution setVariables(String executionId,
      Map<String, Object> variables) {
    SetVariablesCmd cmd = new SetVariablesCmd(executionId);
    cmd.setVariables(variables);
    return commandService.execute(cmd);
  }
}
