/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.wire;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.ow2.orchestra.pvm.env.Context;
import org.ow2.orchestra.pvm.env.Environment;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.util.Closable;
import org.ow2.orchestra.pvm.internal.util.DefaultObservable;

/**
 * object factory that creates, initializes, wires and caches objects based on
 * {@link Descriptor descriptors} (aka IoC container).
 *
 * <h3>General principle</h3>
 *
 * <p>
 * As input, a WireContext takes a {@link WireDefinition}. The WireDefinition
 * contains named {@link Descriptor}s that know how to create objects and wire
 * them together. Each object has a name. The WireContext will maintain a cache
 * (map) of the created objects. So that upon subsequent requests, the same
 * object can be given from the cache.
 * </p>
 *
 * <center><img src="wirescope.gif"/></center>
 *
 * <h3>Purpose</h3>
 *
 * <p>
 * A WireContext is used often in combination with {@link Environment} to
 * decouple the processDefinition virtual machine from its environment. In the
 * {@link org.ow2.orchestra.pvm.env.PvmEnvironmentFactory}, both the environment-factory context and the
 * environment contexts are WireContexts. The PVM will use the persistence
 * service, asynchronous message service, timer service and other services
 * through specified abstractions in the environment.
 * </p>
 *
 * <p>
 * Another usage of the WireContext is construction and configuration of user
 * code objects in a persistable way. {@link org.ow2.orchestra.pvm.activity.Activity}s and
 * {@link org.ow2.orchestra.pvm.activity.ExternalActivity} and other user code can
 * be instantiated with a WireContext. That way, they can be persisted in a
 * fixed schema.
 * </p>
 *
 * <p>
 * Each {@link org.ow2.orchestra.pvm.internal.model.ProcessElementImpl} has configuration properties. Consider this
 * extra metadata that can be associated to elements in a processDefinition
 * definition. In that respect, it's somewhat similar to what annotations are in
 * Java. Because of the wire persistence, all these configuration properties fit
 * into the same process model and in its database schema.
 * </p>
 *
 * <h3>Xml</h3>
 * <p>
 * Mostly often, {@link Descriptor}s and WireContext's are not used directly.
 * Instead, the wire XML is used in a configuration file. The {@link org.ow2.orchestra.pvm.internal.wire.xml.WireParser
 * wire XML parser} contains the documentation on the XML grammer. The
 * {@link org.ow2.orchestra.pvm.internal.wire.xml.WireParser} will produce a {@link WireDefinition} with a bunch of
 * {@link Descriptor}s in it.
 *
 * <h3 id="lifecycle">Object lifecycle</h3>
 *
 * <p>
 * Objects are build in 2 phases: construction and initialization. The
 * motivation for splitting these phases is to resolve many of the circular
 * dependencies. Imagine 2 objects that have a bidirectional reference. By
 * splitting the construction from the initialization phase, the objects can
 * both be constructed first, and then during initialization, they will be
 * injected into each other.
 * </p>
 *
 * <h3>Construction</h3>
 * <p>
 * Construction of the object is all that needs to be done until a reference to
 * the object is available.
 * </p>
 *
 * <p>
 * In the case of dynamically created objects ({@link org.ow2.orchestra.pvm.internal.wire.descriptor.ObjectDescriptor}), the
 * simplest case this is accomplished with a constructor. But also static or
 * non-static factory methods can be used to obtain a reference to an object.
 * </p>
 *
 * <p>
 * In case of immutable objects, the descriptor can just provide a reference to
 * a singleton object.
 * </p>
 *
 * <h3>Initialization</h3>
 * <p>
 * Initialization is optional and it is comprised of everything that needs to be
 * done with an object after a reference to the object is available.
 * {@link org.ow2.orchestra.pvm.internal.wire.descriptor.AbstractDescriptor} contains an empty default initialization method.
 * </p>
 *
 * <p>
 * For objects {@link org.ow2.orchestra.pvm.internal.wire.descriptor.ObjectDescriptor}s, this means that a a sequence of
 * {@link org.ow2.orchestra.pvm.internal.wire.operation.Operation}s can be applied to the object. Following operations
 * implementations are already available and can be applied to an object during
 * initialization:
 * </p>
 *
 * <ul>
 * <li><b>{@link org.ow2.orchestra.pvm.internal.wire.operation.FieldOperation}</b>: injects another object into a field</li>
 * <li><b>{@link org.ow2.orchestra.pvm.internal.wire.operation.PropertyOperation}</b>: injects another object with a setter
 * method.</li>
 * <li><b>{@link org.ow2.orchestra.pvm.internal.wire.operation.InvokeOperation}</b>: invokes a method.</li>
 * <li><b>{@link org.ow2.orchestra.pvm.internal.wire.operation.SubscribeOperation}</b>:
 * subscribes to an {@link org.ow2.orchestra.pvm.internal.util.Observable observable}.</li>
 * </ul>
 *
 * <h3>Environment</h3>
 *
 * <p>
 * When an environment is injected into a WireContext, lookup of all referenced
 * object names will be done first in this WireContext, but if the object name
 * is not defined there, the environment will be searched in the environment's
 * default search order.
 * </p>
 *
 * <h3>Events</h3>
 * <p>
 * Several objects will fire events to which can be subscribed:
 * </p>
 *
 * <p>
 * The WireContext itself fires the {@link #EVENT_OPEN} and {@link #EVENT_OPEN}
 * events.
 * </p>
 *
 * <p>
 * The {@link Descriptor}s will fire the events
 * {@link Descriptor#EVENT_CONSTRUCTING}, {@link Descriptor#EVENT_INITIALIZING},
 * {@link Descriptor#EVENT_CONSTRUCTED}, {@link Descriptor#EVENT_SET} and
 * {@link Descriptor#EVENT_REMOVE}.
 * </p>
 *
 * <p>
 * And last but not least, the objects created by the WireContext can be
 * {@link org.ow2.orchestra.pvm.internal.util.Observable} themselves.
 * </p>
 *
 * <h3>Eager initialization</h3>
 *
 * <p>
 * By default, all objects in a WireContext are lazily constructued and
 * initialized. Eager initialization is specified on a named object and it means
 * that the object is constructed and initialized during construction of the
 * WireContext. You an only specify eager initialization when the object has a
 * name.
 * </p>
 *
 * <h3>Specifying how an object should be initialized.</h3>
 *
 * <p>
 * The initialization can be specified with the
 * {@link org.ow2.orchestra.pvm.internal.wire.descriptor.AbstractDescriptor#setInit(char)} method.
 * </p>
 * The possible value for <code>init</code> parameter is one of :
 * <ul>
 * <li>{@link org.ow2.orchestra.pvm.internal.wire.descriptor.AbstractDescriptor#INIT_LAZY}: for lazy creation and delayed
 * initialization</li>
 * <li>{@link org.ow2.orchestra.pvm.internal.wire.descriptor.AbstractDescriptor#INIT_REQUIRED}: for lazy creation and immediate
 * initialization</li>
 * <li>{@link org.ow2.orchestra.pvm.internal.wire.descriptor.AbstractDescriptor#INIT_EAGER}: for eager creation and delayed
 * initialization</li>
 * <li>{@link org.ow2.orchestra.pvm.internal.wire.descriptor.AbstractDescriptor#INIT_IMMEDIATE}: for eager creation and
 * immediate initialization</li>
 * </ul>
 *
 * @author Tom Baeyens
 * @author Guillaume Porcher (documentation)
 */
public class WireContext extends DefaultObservable implements Context,
    Closable, Serializable {

  private static final long serialVersionUID = 1L;
  private static Log log = Log.getLog(WireContext.class.getName());

  // events ///////////////////////////////////////////////////////////////////

  /**
   * is fired when a new wiring environment is being opened. No event info
   * provided.
   */
  public static final String EVENT_OPEN = "open";
  /**
   * is fired when the wiring environment is being closed. No event info
   * provided.
   */
  public static final String EVENT_CLOSE = "close";

  // member fields ////////////////////////////////////////////////////////////

  protected String name = "wire-context";
  protected transient ClassLoader classLoader;
  protected WireDefinition wireDefinition;

  /** objects that are being instantiated or constructed */
  private Set<String> underConstruction = null;

  /**
   * objects that are constructed, but waiting for the initialization operations
   * (like e.g. injections) to be performed
   */
  private Map<String, PendingInitialization> pendingInitializations = null;

  /**
   * objects on which the initialization operations (like e.g. injections) are
   * being performed
   */
  private Map<String, Object> underInitialization = null;

  /** fully created and initialized objects */
  private Map<String, Object> cache = null;

  /** exceptions throw by descriptor invocations */
  private Map<String, Exception> exceptions = null;

  public WireContext() {
  }

  public WireContext(final WireDefinition wireDefinition) {
    this(wireDefinition, null, null, false);
  }

  /**
   * when this {@link Context} is used in an {@link Environment}, it needs a
   * name.
   */
  public WireContext(final WireDefinition wireDefinition, final String name) {
    this.wireDefinition = wireDefinition;
    this.name = name;
    this.classLoader = wireDefinition != null ? wireDefinition.getClassLoader() : null;
    this.create();
  }

  /**
   * allows for postponing the creation of this wire context.
   *
   * @param delayCreate
   *          specifies if creation should be postponed till {@link #create()}
   *          is called explicitly. If delayCreate is set to false, creation is
   *          done as part of the constructor. If delayCreate is set to true,
   *          the {@link #create()} method needs to be called explicitly by the
   *          client after construction is complete. The use case is creation of
   *          environment where the transactionName needs to be set and the
   *          scope needs to be added to the environment before the creation of
   *          this wire scope is done.
   * @see org.ow2.orchestra.pvm.env.PvmEnvironmentFactory#openEnvironment()
   */
  public WireContext(final WireDefinition wireDefinition, final String name,
      final Environment environment, final boolean delayCreate) {
    this.wireDefinition = wireDefinition;
    this.name = name;
    this.classLoader = wireDefinition != null ? wireDefinition.getClassLoader() : null;

    if (!delayCreate) {
      this.create();
    }
  }

  /** convenience method that wires the object for a given descriptor. */
  public static Object create(final Descriptor descriptor) {
    final WireContext wireContext = new WireContext();
    return wireContext.create(descriptor, false);
  }

  /**
   * initializes the eager objects and then fires the create event. This method
   * only needs to be called explicitly in case <code>delayCreate</code> is true
   * in {@link #WireContext(WireDefinition, String, Environment, boolean)}.
   */
  public void create() {
    WireContext.log.trace("creating " + this.name);
    this.initializeEagerObjects();
    this.fire(WireContext.EVENT_OPEN, null);
  }

  /**
   * Initializes all the eager objects defined in the {@link #wireDefinition}.
   */
  void initializeEagerObjects() {
    if (this.wireDefinition != null) {
      final List<String> eagerInitObjectNames = this.wireDefinition.getEagerInitNames();
      if (eagerInitObjectNames != null) {
        for (final String eagerInitObjectName : eagerInitObjectNames) {
          final Descriptor descriptor = this.wireDefinition
              .getDescriptor(eagerInitObjectName);
          if (descriptor.isEagerInit()) {
            WireContext.log.debug("eagerly initializing " + eagerInitObjectName);
            this.get(eagerInitObjectName, descriptor.isDelayable());
          }
        }
        while ((!this.hasObjectUnderConstruction())
            && (!this.hasObjectUnderInitialization())
            && (this.hasPendingInitializations())) {
          this.processPendingInitializations();
        }
      }
    }
  }

  @Override
  public String toString() {
    return this.name != null ? this.name : super.toString();
  }

  // environment methods
  // //////////////////////////////////////////////////////////

  /**
   * the list of object names defined in this context. This means the union of
   * the object names that are defined in the {@link #wireDefinition} and the
   * objects that are just {@link #set(String, Object)}. If there are no keys,
   * an empty set will be returned.
   */
  public Set<String> keys() {
    final Set<String> keys = new HashSet<String>();
    if (this.cache != null) {
      keys.addAll(this.cache.keySet());
    }
    if (this.wireDefinition != null) {
      final Map<String, Descriptor> descriptors = this.wireDefinition.getDescriptors();
      if (descriptors != null) {
        keys.addAll(descriptors.keySet());
      }
    }
    return keys;
  }

  /**
   * checks if the given objectName is defined, either by means of a descriptor
   * or by an explicit {@link #set(String, Object)}.
   */
  public boolean has(final String objectName) {
    return this.hasCached(objectName) || (this.wireDefinition != null ? this.wireDefinition.hasDescriptor(objectName) : false);
  }

  /**
   * retrieves the object for the given objectName, ensuring it is constructed
   * and initialized.
   *
   * @return the object found, or null if the object was not found.
   */
  public Object get(final String objectName) {
    return this.get(objectName, false);
  }

  /**
   * adds an object to this context, which means storing it in the cache. This
   * doesn't have to be an object that is defined by the {@link WireDefinition}.
   * If an object is set under a certain objectName that also is associated with
   * a descriptor, the object provided in this set invocation will be delivered
   * upon subsequent {@link #get(String)} requests.
   *
   * @return previous value of the object with the name objectName in the
   *         {@link #cache}
   * @throws WireException
   *           when the objectName is null
   */
  public synchronized Object set(final String objectName, final Object object) {
    if (objectName == null) {
      throw new WireException("objectName is null");
    }
    if (this.cache == null) {
      this.cache = new HashMap<String, Object>();
    }
    this.fireObjectEvent(Descriptor.EVENT_SET, objectName, object);
    return this.cache.put(objectName, object);
  }

  /**
   * removes an object from the context and fires the remove event.
   *
   * @return previous object associated with the given name, or null if there
   *         was no mapping for this name.
   */
  public Object remove(final String objectName) {
    Object removed = null;
    if (this.cache != null) {
      removed = this.cache.remove(objectName);
      this.fireObjectEvent(Descriptor.EVENT_REMOVE, objectName, removed);
    }
    return removed;
  }

  /** clears the {@link #cache}. */
  public synchronized void clear() {
    if (this.cache != null) {
      final Set<String> objectsInCache = new HashSet<String>(this.cache.keySet());
      for (final String object : objectsInCache) {
        this.remove(object);
      }
    }
  }

  /**
   * fires the close event then removes the listeners, and cleans up the
   * constructed objects of the context (cleans up the object in the cache and
   * the object in construction).
   *
   * @see #EVENT_CLOSE
   */
  public synchronized void close() {
    WireContext.log.trace("closing " + this.name + "...");

    // fire the close event
    this.fire(WireContext.EVENT_CLOSE, null);
  }

  // object access helper methods /////////////////////////////////////////////

  /**
   * gets the object having the name <code>objectName</code> in this context.
   *
   * @param isDelayable
   *          indicates wether initialization is delayable. When isDelayable is
   *          set to false the returned object will be constructed and
   *          initialized. When isDelayable is set to true, the returned object
   *          will be constructed, but not necessarily initialized.
   * @return the object found or created, or null if the object was not found
   *         and cannot be created.
   * @throws WireException
   *           if a circular dependency was found during the object creation.
   */
  public synchronized Object get(final String objectName, final boolean isDelayable) {
    if (this.hasException(objectName)) {
      throw new WireException("getting " + objectName
          + " previously resulted in an exception", this.exceptions.get(objectName));
    }

    // first check if the object is in the cache
    if (this.hasCached(objectName)) {
      final Object object = this.cache.get(objectName);
      WireContext.log.trace("delivering " + objectName);
      return object;
    }

    // then check if it is constructed, but not yet in the cache (pending or
    // under initialization)
    final Object constructed = this.getConstructed(objectName);
    if (isDelayable && (null != constructed)) {
      final Object object = constructed;
      WireContext.log.trace("providing already constructed " + objectName);
      return object;
    }

    // then check if we can create the object from a descriptor
    if (this.wireDefinition.hasDescriptor(objectName)) {

      if (this.isUnderConstruction(objectName) || this.isUnderInitialization(objectName)) {
        throw new WireException("circular dependency for " + objectName);
      }

      return this.create(objectName, isDelayable);
    }

    // then check if we can find it in the environment (if one is available)
    final Environment environment = Environment.getCurrent();
    if (environment != null) {
      WireContext.log.trace("delivering " + objectName + " from environment");
      return environment.get(objectName);
    }

    WireContext.log.trace("delivering null for undefined object " + objectName);
    return null;
  }

  /**
   * creates a new object for the given objectName as defined in the
   * {@link #wireDefinition}.
   *
   * @param isDelayable
   *          indicates wether initialization is delayable. When isDelayable is
   *          set to false the returned object will be constructed and
   *          initialized. When isDelayable is set to true, the returned object
   *          will be constructed, but not necessarily initialized.
   */
  protected Object create(final String objectName, final boolean isDelayable) {
    final Descriptor descriptor = this.wireDefinition.getDescriptor(objectName);
    return this.create(descriptor, isDelayable);
  }

  /**
   * creates a new object for the given descriptor.
   *
   * @param isDelayable
   *          indicates wether initialization is delayable. When isDelayable is
   *          set to false the returned object will be constructed and
   *          initialized. When isDelayable is set to true, the returned object
   *          will be constructed, but not necessarily initialized.
   */
  public Object create(final Descriptor descriptor, final boolean isDelayable) {
    Object object = null;

    object = this.construct(descriptor);
    this.initialize(object, descriptor, isDelayable);
    this.processPendingInitializations();

    return object;
  }

  Object construct(final Descriptor descriptor) {
    Object object;

    final String objectName = descriptor.getName();
    if (objectName != null) {
      this.fireObjectEvent(Descriptor.EVENT_CONSTRUCTING, objectName, null);
      if (this.underConstruction == null) {
        this.underConstruction = new HashSet<String>();
      }
      this.underConstruction.add(objectName);
      WireContext.log.trace("constructing " + objectName);
    }

    try {
      object = descriptor.construct(this);
    } catch (final RuntimeException e) {
      this.addException(descriptor, e);
      throw e;
    }

    if (objectName != null) {
      this.underConstruction.remove(objectName);
    }

    return object;
  }

  // initialization ///////////////////////////////////////////////////////////

  private enum InitializationType {
    NONE, IMMEDIATE, DELAYEBLE
  }

  void initialize(final Object object, final Descriptor descriptor, final boolean isDelayable) {

    final InitializationType initializationType = this.getInitializationType(object,
        descriptor, isDelayable);

    if (initializationType == InitializationType.IMMEDIATE) {
      this.performInitialization(object, descriptor);

    } else if (initializationType == InitializationType.DELAYEBLE) {
      this.addPendingInitialization(object, descriptor);

    } else {
      final String objectName = descriptor.getName();
      if (objectName != null) {
        this.set(objectName, object);
      }
    }
  }

  InitializationType getInitializationType(final Object object,
      final Descriptor descriptor, final boolean isDelayable) {
    if (object == null) {
      return InitializationType.NONE;
    }

    if (isDelayable && descriptor.isDelayable()) {
      return InitializationType.DELAYEBLE;
    }

    return InitializationType.IMMEDIATE;
  }

  void performInitialization(final Object object, final Descriptor descriptor) {
    final String objectName = descriptor.getName();

    if (objectName != null) {
      this.fireObjectEvent(Descriptor.EVENT_INITIALIZING, objectName, object);
      if (this.underInitialization == null) {
        this.underInitialization = new HashMap<String, Object>();
      }
      this.underInitialization.put(objectName, object);
      WireContext.log.trace("initializing " + objectName);
    }

    try {
      descriptor.initialize(object, this);
    } catch (final RuntimeException e) {
      this.addException(descriptor, e);
      throw e;
    }

    if (objectName != null) {
      this.underInitialization.remove(objectName);
      // event constructed is fired before the object is put in the cache
      // because that generates a set event
      this.fireObjectEvent(Descriptor.EVENT_CONSTRUCTED, objectName, object);
      this.set(objectName, object);
    }
  }

  void addPendingInitialization(final Object object, final Descriptor descriptor) {
    if (this.pendingInitializations == null) {
      this.pendingInitializations = new HashMap<String, PendingInitialization>();
    }
    this.pendingInitializations.put(descriptor.getName(), new PendingInitialization(
        object, descriptor));
  }

  void processPendingInitializations() {
    if (this.pendingInitializations != null) {
      final Collection<PendingInitialization> pendingInitializationValues = new HashSet<PendingInitialization>(
          this.pendingInitializations.values());
      for (PendingInitialization pi : pendingInitializationValues) {
        // move pi from pending initializations to under initialization
        final String objectName = pi.initializable.getName();
        pi = this.pendingInitializations.remove(objectName);
        if (pi != null) {
          // initialize
          this.performInitialization(pi.object, pi.initializable);
        }
      }
    }
  }

  boolean hasPendingInitializations() {
    return (this.pendingInitializations != null) && (!this.pendingInitializations.isEmpty());
  }

  /**
   * container for an storing waiting objects and their initializable in the
   * list {@link #pendingInitializations}, while waiting for initialization.
   */
  class PendingInitialization {
    private final Object object;
    private final Descriptor initializable;

    public PendingInitialization(final Object object, final Descriptor descriptor) {
      this.object = object;
      this.initializable = descriptor;
    }

    @Override
    public String toString() {
      final String objectName = this.initializable.getName();
      return "PendingInitialization["
          + (objectName != null ? objectName + "|" : "") + this.object + "]";
    }
  }

  /**
   * checks if the given objectName is available in the cache, which means it
   * already has been constructed from a wire definition or it has been
   * {@link #set(String, Object)} explicitely.
   */
  public boolean hasCached(final String objectName) {
    return (this.cache != null) && (this.cache.containsKey(objectName));
  }

  /** finds the object in all stages after construction. */
  Object getConstructed(final String objectName) {
    Object constructed = null;
    if ((this.pendingInitializations != null)
        && (this.pendingInitializations.containsKey(objectName))) {
      constructed = this.pendingInitializations.get(objectName).object;
    } else if ((this.underInitialization != null)
        && (this.underInitialization.containsKey(objectName))) {
      constructed = this.underInitialization.get(objectName);
    }
    return constructed;
  }

  /** fires a {@link WireObjectEventInfo}. */
  protected void fireObjectEvent(final String eventName, final String objectName,
      final Object object) {
    WireObjectEventInfo wireEvent = null;

    // first fire the event on the descriptor for object specific listeners
    if (this.wireDefinition != null) {
      final Map<String, Descriptor> descriptors = this.wireDefinition.getDescriptors();
      if (descriptors != null) {
        final Descriptor descriptor = descriptors.get(objectName);
        if (descriptor != null) {
          if (wireEvent == null) {
            wireEvent = new WireObjectEventInfo(eventName, objectName, object);
          }
          descriptor.fire(eventName, wireEvent);
        }
      }
    }

    // then fire the event on this wiring environment for global listeners
    if ((this.listeners != null) && (wireEvent == null)) {
      wireEvent = new WireObjectEventInfo(eventName, objectName, object);
    }

    this.fire(eventName, wireEvent);
  }

  boolean hasObjectUnderConstruction() {
    return (this.underConstruction != null) && (!this.underConstruction.isEmpty());
  }

  boolean hasObjectUnderInitialization() {
    return (this.underInitialization != null) && (!this.underInitialization.isEmpty());
  }

  boolean isUnderConstruction(final String objectName) {
    return (this.underConstruction != null) && (this.underConstruction.contains(objectName));
  }

  boolean isUnderInitialization(final String objectName) {
    return (this.underInitialization != null) && (this.underInitialization.containsKey(objectName));
  }

  /**
   * the class loader to use to create objects or if none was explicitely set in
   * this wire context, the default context class loader for the current thread.
   */
  public ClassLoader getClassLoader() {
    // if there is a specific classloader specified
    if (this.classLoader != null) {
      return this.classLoader;
    }
    // otherwise, use the current thread classloader
    return Thread.currentThread().getContextClassLoader();
  }

  public boolean hasClassLoader() {
    return this.classLoader != null;
  }

  // search by class //////////////////////////////////////////////////////////

  /**
   * searches for the first descriptor that defines an object of the given type.
   * In case of multiple objects of the same type, the first object that is
   * declared of the given type will be found. Also super classes and interfaces
   * are taken into account. Not all descriptor types will be type sensitive,
   * only:
   *
   * <pre>
   * | ObjectDescriptor                       | object            |
   * | HibernatePersistenceServiceDescriptor  | business-calendar |
   * | TransactionDescriptor                  | transaction       |
   * | PropertiesDescriptor                   | properties        |
   * | BusinessCalendarDescriptor             | business-calendar |
   * &lt;/ul&gt;
   * </pre>
   */
  @SuppressWarnings("unchecked")
  public <T> T get(final Class<T> type) {
    if (this.wireDefinition != null) {
      final String descrName = this.wireDefinition.getDescriptorName(type);
      if (descrName != null) {
        return (T) this.get(descrName);
      }
    }
    return null;
  }

  protected boolean hasException(final String objectName) {
    return (this.exceptions != null) && (this.exceptions.containsKey(objectName));
  }

  protected void addException(final Descriptor descriptor, final Exception exception) {
    if (this.exceptions == null) {
      this.exceptions = new HashMap<String, Exception>();
    }
    this.exceptions.put(descriptor.getName(), exception);
  }

  // getters and setters //////////////////////////////////////////////////////

  public String getName() {
    return this.name;
  }

  public void setClassLoader(final ClassLoader classLoader) {
    this.classLoader = classLoader;
  }

  public WireDefinition getWireDefinition() {
    return this.wireDefinition;
  }

  public void setWireDefinition(final WireDefinition wireDefinition) {
    this.wireDefinition = wireDefinition;
  }
}
