package org.ow2.orchestra.pvm.internal.wire.binding;

import java.util.ArrayList;
import java.util.List;

import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.internal.util.PvmXmlUtil;
import org.ow2.orchestra.pvm.internal.util.ReflectUtil;
import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.CollectionDescriptor;
import org.ow2.orchestra.pvm.internal.wire.xml.WireParser;
import org.ow2.orchestra.pvm.internal.xml.Parse;
import org.ow2.orchestra.pvm.internal.xml.Parser;
import org.ow2.orchestra.util.XmlUtil;
import org.w3c.dom.Element;

public abstract class AbstractCollectionBinding extends WireDescriptorBinding {

  public AbstractCollectionBinding(final String tagName) {
    super(tagName);
  }

  public Object parse(final Element element, final Parse parse, final Parser parser) {
    final CollectionDescriptor descriptor = this.createDescriptor();

    final String className = XmlUtil.attribute(element, "class");

    // verify if the given classname is specified and implements the collection
    // interface
    if (AbstractCollectionBinding.verify(className, this.getCollectionInterface(), parse, parser)) {
      descriptor.setClassName(className);
    }

    final Boolean isSynchronized = PvmXmlUtil.attributeBoolean(element, "synchronized",
        false, parse);
    if (isSynchronized != null) {
      descriptor.setSynchronized(isSynchronized.booleanValue());
    }

    final List<Descriptor> valueDescriptors = new ArrayList<Descriptor>();
    final List<Element> elements = XmlUtil.elements(element);
    if (elements != null) {
      for (final Element valueElement : elements) {
        final Descriptor valueDescriptor = (Descriptor) parser.parseElement(
            valueElement, parse, WireParser.CATEGORY_DESCRIPTOR);
        if (valueDescriptor != null) {
          valueDescriptors.add(valueDescriptor);
        }
      }
    }
    descriptor.setValueDescriptors(valueDescriptors);
    return descriptor;
  }

  /**
   * verifies if the given classname is specified and implements the collection
   * interface
   */
  public static boolean verify(final String className, final Class< ? > collectionInterface,
      final Parse parse, final Parser parser) {
    if (className == null) {
      return false;
    }

    try {
      final Class< ? > collectionClass = ReflectUtil.loadClass(parse.getClassLoader(), className);

      if (collectionInterface.isAssignableFrom(collectionClass)) {
        return true;
      } else {
        parse.addProblem("class " + className + " is not a "
            + collectionInterface.getName());
      }
    } catch (final PvmException e) {
      parse.addProblem("class " + className + " could not be found");
    }
    return false;
  }

  protected abstract Class< ? > getCollectionInterface();

  protected abstract CollectionDescriptor createDescriptor();
}
