/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.wire.binding;

import org.ow2.orchestra.pvm.internal.cmd.CommandService;
import org.ow2.orchestra.pvm.internal.jobexecutor.DeadJobHandler;
import org.ow2.orchestra.pvm.internal.jobexecutor.jdk.JdkJobExecutor;
import org.ow2.orchestra.pvm.internal.jobexecutor.pvm.PvmJobExecutor;
import org.ow2.orchestra.pvm.internal.util.PvmXmlUtil;
import org.ow2.orchestra.pvm.internal.wire.descriptor.ContextTypeRefDescriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.IntegerDescriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.JobExecutorDescriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.ObjectDescriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.ReferenceDescriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.StringDescriptor;
import org.ow2.orchestra.pvm.internal.wire.operation.InvokeOperation;
import org.ow2.orchestra.pvm.internal.xml.Parse;
import org.ow2.orchestra.pvm.internal.xml.Parser;
import org.w3c.dom.Element;

/**
 * parses a descriptor for creating a {@link org.ow2.orchestra.pvm.internal.jobexecutor.JobExecutor}.
 *
 * See schema docs for more details.
 *
 * @author Tom Baeyens
 * @author Pascal Verdage
 */
public class JobExecutorBinding extends WireDescriptorBinding {

  public JobExecutorBinding() {
    super("job-executor");
  }

  public Object parse(final Element element, final Parse parse, final Parser parser) {
    String type = "default";
    if (element.hasAttribute("type")) {
      type = element.getAttribute("type");
    }

    JobExecutorDescriptor descriptor = null;
    if (type.equals("default")) {
      descriptor = new JobExecutorDescriptor(PvmJobExecutor.class);

      this.parseIntAttribute(element, "idle-max", descriptor, "idleMillisMax", parse);
      this.parseIntAttribute(element, "history", descriptor, "historySize", parse);
      this.parseIntAttribute(element, "threads", descriptor, "nbrOfThreads", parse);

    } else if (type.equals("jdk")) {
      descriptor = new JobExecutorDescriptor(JdkJobExecutor.class);
    } else {
      parse.addProblem("JobExecutor type: " + type + " is not supported.");
      return null;
    }

    // add common attributes


    // inject the command executor

    if (element.hasAttribute("command-service")) {
      descriptor.addInjection("commandService", new ReferenceDescriptor(element
          .getAttribute("command-service")));
    } else {
      descriptor.addInjection("commandService", new ContextTypeRefDescriptor(
          CommandService.class));
    }

    // inject the dead job handler executor

    if (element.hasAttribute("dead-job-handler")) {
      descriptor.addInjection("deadJobHandler", new ReferenceDescriptor(element
          .getAttribute("dead-job-handler")));
    } else {
      descriptor.addInjection("deadJobHandler", new ContextTypeRefDescriptor(
          DeadJobHandler.class));
    }

    if (element.hasAttribute("name")) {
      descriptor.addInjection("name", new StringDescriptor(element
          .getAttribute("name")));
    }

    this.parseIntAttribute(element, "idle", descriptor, "idleMillis", parse);

    this.parseIntAttribute(element, "lock", descriptor, "lockMillis", parse);

    // by default invoke the start method, unless auto-start is disabled
    if (PvmXmlUtil.attributeBoolean(element, "auto-start", false, parse, true)) {
      final InvokeOperation invokeStartOperation = new InvokeOperation();
      invokeStartOperation.setMethodName("start");
      descriptor.addOperation(invokeStartOperation);
      descriptor.setAutoStart(true);
    }

    return descriptor;
  }

  private void parseIntAttribute(final Element element, final String attributeName,
      final ObjectDescriptor descriptor, final String fieldName, final Parse parse) {
    final Integer intValue = PvmXmlUtil.attributeInteger(element, attributeName, false,
        parse);
    if (intValue != null) {
      descriptor.addInjection(fieldName, new IntegerDescriptor(intValue));
    }
  }
}
