/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.wire.descriptor;

import java.net.URL;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

import org.hibernate.JDBCException;
import org.hibernate.cfg.Configuration;
import org.hibernate.connection.ConnectionProvider;
import org.hibernate.connection.ConnectionProviderFactory;
import org.hibernate.dialect.Dialect;
import org.hibernate.tool.hbm2ddl.DatabaseMetadata;
import org.hibernate.util.JDBCExceptionReporter;
import org.ow2.orchestra.pvm.PvmException;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.util.ReflectUtil;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireDefinition;
import org.ow2.orchestra.pvm.internal.wire.WireException;
import org.ow2.orchestra.pvm.internal.wire.operation.Operation;

/**
 * @author Tom Baeyens
 */
public class HibernateConfigurationDescriptor extends AbstractDescriptor {

  private static final long serialVersionUID = 1L;

  private static final Log LOG = Log.getLog(HibernateConfigurationDescriptor.class.getName());

  private String className;
  private String namingStrategyClassName;
  private List<Operation> mappingOperations;
  private List<Operation> cacheOperations;
  private PropertiesDescriptor propertiesDescriptor;
  private Operation schemaOperation;

  public Object construct(final WireContext wireContext) {
    // instantiation of the configuration
    Configuration configuration = null;
    if (this.className != null) {
      final ClassLoader classLoader = wireContext.getClassLoader();
      HibernateConfigurationDescriptor.LOG.trace("instantiating hibernate configation "
          + this.name + " with class " + this.className);
      final Class< ? > configurationClass = ReflectUtil.loadClass(classLoader, this.className);
      configuration = (Configuration) ReflectUtil.newInstance(configurationClass);
    } else {
      HibernateConfigurationDescriptor.LOG.trace("instantiating default hibernate configuration  " + this.name);
      configuration = new Configuration();
    }
    return configuration;
  }

  @Override
  public void initialize(final Object object, final WireContext wireContext) {
    final Configuration configuration = (Configuration) object;
    this.apply(this.mappingOperations, configuration, wireContext);
    this.apply(this.cacheOperations, configuration, wireContext);
    if (this.propertiesDescriptor != null) {
      final Properties properties = (Properties) wireContext.create(
          this.propertiesDescriptor, false);
      if (HibernateConfigurationDescriptor.LOG.isDebugEnabled()) {
        HibernateConfigurationDescriptor.LOG
            .debug("adding properties to hibernate configuration " + this.name + ": "
                + properties);
      }
      configuration.addProperties(properties);
    }
    if (this.schemaOperation != null) {
      this.schemaOperation.apply(configuration, wireContext);
    }
  }

  private void apply(final List<Operation> operations, final Configuration configuration,
      final WireContext wireContext) {
    if (operations != null) {
      for (final Operation operation : operations) {
        HibernateConfigurationDescriptor.LOG.trace("hibernate configuration " + this.name + ": " + operation.toString());
        operation.apply(configuration, wireContext);
      }
    }
  }

  @Override
  public Class< ? > getType(final WireDefinition wireDefinition) {
    if (this.className != null) {
      try {
        return ReflectUtil
            .loadClass(wireDefinition.getClassLoader(), this.className);
      } catch (final PvmException e) {
        throw new WireException("couldn't create hibernate configuration '"
            + this.className + "': " + e.getMessage(), e.getCause());
      }
    }
    return Configuration.class;
  }

  public void addMappingOperation(final Operation operation) {
    if (this.mappingOperations == null) {
      this.mappingOperations = new ArrayList<Operation>();
    }
    this.mappingOperations.add(operation);
  }

  public void addCacheOperation(final Operation operation) {
    if (this.cacheOperations == null) {
      this.cacheOperations = new ArrayList<Operation>();
    }
    this.cacheOperations.add(operation);
  }

  // operations ///////////////////////////////////////////////////////////////

  public static class AddResource implements Operation {
    private static final long serialVersionUID = 1L;
    private final String resource;

    public AddResource(final String resource) {
      this.resource = resource;
    }

    public void apply(final Object target, final WireContext wireContext) {
      final Configuration configuration = (Configuration) target;
      configuration.addResource(this.resource, wireContext.getClassLoader());
    }

    @Override
    public String toString() {
      return "adding mapping resource " + this.resource
          + " to hibernate configuration";
    }
  }

  public static class AddFile implements Operation {
    private static final long serialVersionUID = 1L;
    private final String fileName;

    public AddFile(final String fileName) {
      this.fileName = fileName;
    }

    public void apply(final Object target, final WireContext wireContext) {
      final Configuration configuration = (Configuration) target;
      configuration.addFile(this.fileName);
    }

    @Override
    public String toString() {
      return "adding hibernate mapping file " + this.fileName + " to configuration";
    }
  }

  public static class AddClass implements Operation {
    private static final long serialVersionUID = 1L;
    private final String className;

    public AddClass(final String className) {
      this.className = className;
    }

    public void apply(final Object target, final WireContext wireContext) {
      final Configuration configuration = (Configuration) target;
      try {
        final Class< ? > persistentClass = wireContext.getClassLoader().loadClass(
            this.className);
        configuration.addClass(persistentClass);
      } catch (final Exception e) {
        throw new PvmException("couldn't add mapping for class " + this.className, e);
      }
    }

    @Override
    public String toString() {
      return "adding persistent class " + this.className
          + " to hibernate configuration";
    }
  }

  public static class AddUrl implements Operation {
    private static final long serialVersionUID = 1L;
    private final String url;

    public AddUrl(final String url) {
      this.url = url;
    }

    public void apply(final Object target, final WireContext wireContext) {
      final Configuration configuration = (Configuration) target;
      try {
        configuration.addURL(new URL(this.url));
      } catch (final Exception e) {
        throw new PvmException(
            "couldn't add hibernate mapping from URL " + this.url, e);
      }
    }
  }

  public static class SetCacheConcurrencyStrategy implements Operation {
    private static final long serialVersionUID = 1L;
    private final String className;
    private final String concurrencyStrategy;

    public SetCacheConcurrencyStrategy(final String className,
        final String concurrencyStrategy) {
      this.className = className;
      this.concurrencyStrategy = concurrencyStrategy;
    }

    public void apply(final Object target, final WireContext wireContext) {
      final Configuration configuration = (Configuration) target;
      configuration.setCacheConcurrencyStrategy(this.className, this.concurrencyStrategy);
    }

    @Override
    public String toString() {
      return "setting cache concurrency strategy on class " + this.className
          + " to " + this.concurrencyStrategy + " on hibernate configuration";
    }
  }

  public static class SetCollectionCacheConcurrencyStrategy implements
      Operation {
    private static final long serialVersionUID = 1L;
    private final String collection;
    private final String concurrencyStrategy;

    public SetCollectionCacheConcurrencyStrategy(final String collection,
        final String concurrencyStrategy) {
      this.collection = collection;
      this.concurrencyStrategy = concurrencyStrategy;
    }

    public void apply(final Object target, final WireContext wireContext) {
      final Configuration configuration = (Configuration) target;
      configuration.setCollectionCacheConcurrencyStrategy(this.collection,
          this.concurrencyStrategy);
    }

    @Override
    public String toString() {
      return "setting cache concurrency strategy on collection " + this.collection
          + " to " + this.concurrencyStrategy + " on hibernate configuration";
    }
  }

  public static final class CreateSchema implements Operation {

    private static final long serialVersionUID = 1L;

    /** The sole instance of this class */
    private static final Operation INSTANCE = new CreateSchema();

    private CreateSchema() {
      // suppress default constructor, ensuring non-instantiability
    }

    public void apply(final Object target, final WireContext wireContext) {
      final Configuration configuration = (Configuration) target;
      final Properties cfgProperties = configuration.getProperties();
      final Dialect dialect = Dialect.getDialect(cfgProperties);
      final ConnectionProvider connectionProvider = ConnectionProviderFactory
          .newConnectionProvider(cfgProperties);
      try {
        final Connection connection = connectionProvider.getConnection();
        try {
          HibernateConfigurationDescriptor.LOG.debug("dropping db schema");
          final String[] dropScript = configuration.generateDropSchemaScript(dialect);
          HibernateConfigurationDescriptor.executeScript(connection, dropScript);
          HibernateConfigurationDescriptor.LOG.debug("creating db schema");
          final String[] createScript = configuration
              .generateSchemaCreationScript(dialect);
          HibernateConfigurationDescriptor.executeScript(connection, createScript);
        } finally {
          connectionProvider.closeConnection(connection);
        }
      } catch (final SQLException e) {
        throw new JDBCException("error creating schema", e);
      } finally {
        connectionProvider.close();
      }
    }

    /** Returns the sole instance of this class */
    public static Operation getInstance() {
      return CreateSchema.INSTANCE;
    }
  }

  public static final class UpdateSchema implements Operation {

    private static final long serialVersionUID = 1L;

    private static final Operation INSTANCE = new UpdateSchema();

    private UpdateSchema() {
      // suppress default constructor, ensuring non-instantiability
    }

    public void apply(final Object target, final WireContext wireContext) {
      final Configuration configuration = (Configuration) target;
      final Properties cfgProperties = configuration.getProperties();
      final Dialect dialect = Dialect.getDialect(cfgProperties);
      final ConnectionProvider connectionProvider = ConnectionProviderFactory
          .newConnectionProvider(cfgProperties);
      try {
        final Connection connection = connectionProvider.getConnection();
        try {
          final DatabaseMetadata metadata = new DatabaseMetadata(connection, dialect);
          final String[] updateScript = configuration.generateSchemaUpdateScript(
              dialect, metadata);
          HibernateConfigurationDescriptor.LOG.debug("updating db schema");
          HibernateConfigurationDescriptor.executeScript(connection, updateScript);
        } finally {
          connectionProvider.closeConnection(connection);
        }
      } catch (final SQLException e) {
        throw new JDBCException("error updating schema", e);
      } finally {
        connectionProvider.close();
      }
    }

    public static Operation getInstance() {
      return UpdateSchema.INSTANCE;
    }
  }

  private static List<SQLException> executeScript(final Connection connection,
      final String[] script) throws SQLException {
    List<SQLException> exceptions = Collections.emptyList();
    final Statement statement = connection.createStatement();
    try {
      for (final String line : script) {
        HibernateConfigurationDescriptor.LOG.debug(line);
        try {
          statement.executeUpdate(line);
          if (statement.getWarnings() != null) {
            JDBCExceptionReporter.logAndClearWarnings(connection);
          }
        } catch (final SQLException e) {
          if (exceptions.isEmpty()) {
            exceptions = new ArrayList<SQLException>();
          }
          exceptions.add(e);
        }
      }
    } finally {
      statement.close();
    }
    return exceptions;
  }

  // getters and setters //////////////////////////////////////////////////////

  public String getClassName() {
    return this.className;
  }

  public void setClassName(final String className) {
    this.className = className;
  }

  public PropertiesDescriptor getPropertiesDescriptor() {
    return this.propertiesDescriptor;
  }

  public void setPropertiesDescriptor(final PropertiesDescriptor propertiesDescriptor) {
    this.propertiesDescriptor = propertiesDescriptor;
  }

  public String getNamingStrategyClassName() {
    return this.namingStrategyClassName;
  }

  public void setNamingStrategyClassName(final String namingStrategyClassName) {
    this.namingStrategyClassName = namingStrategyClassName;
  }

  public Operation getSchemaOperation() {
    return this.schemaOperation;
  }

  public void setSchemaOperation(final Operation schemaOperation) {
    this.schemaOperation = schemaOperation;
  }
}
