/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.ow2.orchestra.pvm.internal.wire.descriptor;

import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.util.Listener;
import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireDefinition;
import org.ow2.orchestra.pvm.internal.wire.WireException;

/**
 * @author Tom Baeyens
 */
public class HibernateSessionFactoryDescriptor extends AbstractDescriptor {

  private static final long serialVersionUID = 1L;
  private static final Log LOG = Log.getLog(HibernateSessionFactoryDescriptor.class.getName());

  private String configurationName;
  private Descriptor configurationDescriptor;

  public Object construct(final WireContext wireContext) {
    Configuration configuration = null;

    if (this.configurationName != null) {
      configuration = (Configuration) wireContext.get(this.configurationName);

    } else if (this.configurationDescriptor != null) {
      configuration = (Configuration) wireContext.create(
          this.configurationDescriptor, false);

    } else {
      configuration = wireContext.get(Configuration.class);
    }

    if (configuration == null) {
      throw new WireException("couldn't find configuration");
    }

    final SessionFactory sessionFactory = configuration.buildSessionFactory();

    wireContext.addListener(new SessionFactoryCloser(sessionFactory));

    return sessionFactory;
  }

  public static class SessionFactoryCloser implements Listener {
    private final SessionFactory sessionFactory;

    public SessionFactoryCloser(final SessionFactory sessionFactory) {
      this.sessionFactory = sessionFactory;
    }

    public void event(final Object source, final String eventName, final Object info) {
      if (WireContext.EVENT_CLOSE.equals(eventName)) {
        HibernateSessionFactoryDescriptor.LOG.trace("closing hibernate session factory");
        this.sessionFactory.close();
      }
    }
  }

  @Override
  public Class< ? > getType(final WireDefinition wireDefinition) {
    return SessionFactory.class;
  }

  public void setConfigurationName(final String configurationName) {
    this.configurationName = configurationName;
  }

  public void setConfigurationDescriptor(final Descriptor configurationDescriptor) {
    this.configurationDescriptor = configurationDescriptor;
  }
}
