package org.ow2.orchestra.pvm.internal.wire.operation;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.List;

import org.ow2.orchestra.pvm.internal.log.Log;
import org.ow2.orchestra.pvm.internal.util.Listener;
import org.ow2.orchestra.pvm.internal.util.ReflectUtil;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireException;
import org.ow2.orchestra.pvm.internal.wire.descriptor.ArgDescriptor;
import org.ow2.orchestra.pvm.internal.wire.descriptor.ObjectDescriptor;

/**
 * Wrapper for the subscribe operation. This class will be used to call a
 * specified method on reception of an event. This class is used so that a non
 * {@link Listener} class can subscribe to an {@link org.ow2.orchestra.pvm.internal.util.Observable} object.
 *
 * @see SubscribeOperation
 *
 * @author Tom Baeyens
 * @author Guillaume Porcher (Documentation)
 */
public class MethodInvokerListener implements Listener, Serializable {

  private static final long serialVersionUID = 1L;
  private static Log log = Log.getLog(MethodInvokerListener.class.getName());

  private final String methodName;
  private List<ArgDescriptor> argDescriptors = null;
  private final WireContext wireContext;
  private final Object target;

  /**
   * Creates a new Wrapper. When an event is received, the arguments
   * <code>args</code> are created from the list <code>argDescriptors</code>,
   * and <code>target.methodName(args)</code> is called.
   *
   * @param methodName
   *          name of the method to call when an event is received.
   * @param argDescriptors
   *          list of descriptors of arguments given to the method.
   * @param wireContext
   *          context to use to create the arguments
   * @param target
   *          object on which the method will be called.
   */
  public MethodInvokerListener(final String methodName,
      final List<ArgDescriptor> argDescriptors, final WireContext wireContext, final Object target) {
    this.methodName = methodName;
    this.argDescriptors = argDescriptors;
    this.wireContext = wireContext;
    this.target = target;
  }

  public void event(final Object source, final String eventName, final Object info) {
    MethodInvokerListener.log.debug("invoking " + this.methodName + " on " + this.target + " for event "
        + eventName);
    try {
      final Object[] args = ObjectDescriptor.getArgs(this.wireContext, this.argDescriptors);
      final Class< ? > clazz = this.target.getClass();
      final Method method = ReflectUtil.findMethod(clazz, this.methodName, this.argDescriptors,
          args);
      if (method == null) {
        throw new WireException("method "
            + ReflectUtil.getSignature(this.methodName, this.argDescriptors, args)
            + " unavailable for " + this.target);
      }
      ReflectUtil.invoke(method, this.target, args);
    } catch (final WireException e) {
      throw e;
    } catch (final Exception e) {
      throw new WireException("couldn't invoke listener method " + this.methodName
          + ": " + e.getMessage(), e);
    }
  }
}
