package org.ow2.orchestra.pvm.internal.wire.operation;

import java.lang.reflect.Method;

import org.ow2.orchestra.pvm.internal.util.ReflectUtil;
import org.ow2.orchestra.pvm.internal.wire.Descriptor;
import org.ow2.orchestra.pvm.internal.wire.WireContext;
import org.ow2.orchestra.pvm.internal.wire.WireException;

/**
 * injects another object with a setter method.
 *
 * @author Tom Baeyens
 * @author Guillaume Porcher (documentation)
 *
 */
public class PropertyOperation extends AbstractOperation {

  private static final long serialVersionUID = 1L;

  private String setterName = null;
  /* the method will be searched by reflection on the runtime value */
  private Descriptor descriptor = null;

  public void apply(final Object target, final WireContext wireContext) {
    // create the value to assign to the property
    final Object value = wireContext.create(this.descriptor, true);
    Method method = null;
    final Class< ? > clazz = target.getClass();
    final Object[] args = new Object[] { value };
    method = ReflectUtil.findMethod(clazz, this.setterName, null, args);
    if (method == null) {
      throw new WireException("couldn't find property setter " + this.setterName
          + " for value " + value);
    }
    ReflectUtil.invoke(method, target, args);
  }

  /**
   * Sets the name of the property that should be updated by this operation. If
   * propertyName is <code>foo</code>, the method used to set the value will be
   * <code>setFoo</code>.
   *
   * @param propertyName
   */
  public void setPropertyName(final String propertyName) {
    this.setterName = "set" + propertyName.substring(0, 1).toUpperCase()
        + propertyName.substring(1);
  }

  /**
   * Gets the name of the setter method used to inject the property value.
   *
   * @return name of the setter method used to inject the property value.
   */
  public String getSetterName() {
    return this.setterName;
  }

  /**
   * Sets the name of the setter method to use to inject the property value.
   *
   * @param setterName
   */
  public void setSetterName(final String setterName) {
    this.setterName = setterName;
  }

  /**
   * Gets the descriptor used to create the field's value.
   */
  public Descriptor getDescriptor() {
    return this.descriptor;
  }

  /**
   * Sets the descriptor used to create the field's value
   *
   * @param valueDescriptor
   */
  public void setDescriptor(final Descriptor valueDescriptor) {
    this.descriptor = valueDescriptor;
  }
}
